const fs = require('fs');
const path = require('path');

// Primary env var name for the app install root; keep an alias for compatibility.
const INSTALL_ENV_VARS = ['HAM_HOME', 'AGENTGARDEN_HOME'];

function pickEnvVar() {
  return INSTALL_ENV_VARS.find((key) => {
    const val = process.env[key];
    return typeof val === 'string' && val.trim();
  });
}

function resolveInstallRoot() {
  const fromEnv = pickEnvVar();
  if (fromEnv) {
    return path.resolve(process.env[fromEnv]);
  }

  // Packaged Electron apps place resources under process.resourcesPath.
  if (typeof process.resourcesPath === 'string') {
    const appDir = path.join(process.resourcesPath, 'app');
    const appAsar = `${appDir}.asar`;
    if (fs.existsSync(appDir)) return appDir;
    if (fs.existsSync(appAsar)) return appAsar;
  }

  // Fallback to the directory containing this helper (repo root in dev).
  return path.resolve(__dirname);
}

function ensureInstallEnv() {
  const installRoot = resolveInstallRoot();
  const active = pickEnvVar() || INSTALL_ENV_VARS[0];
  INSTALL_ENV_VARS.forEach((name) => {
    if (!process.env[name]) {
      process.env[name] = installRoot;
    }
  });
  return { envVar: active, installRoot };
}

function pathFromInstall(...segments) {
  const { installRoot } = ensureInstallEnv();
  return path.join(installRoot, ...segments);
}

function getInstallEnvSummary() {
  const { envVar, installRoot } = ensureInstallEnv();
  return { envVar, installRoot };
}

module.exports = {
  INSTALL_ENV_VARS,
  ensureInstallEnv,
  getInstallEnvSummary,
  pathFromInstall,
  resolveInstallRoot,
};
