#!/usr/bin/env node
/**
 * Report sub-agent status by inspecting temp files.
 *
 * Usage: node agent-status.js --master <id> [--stale-minutes 10] [--json] [--limit N] [--out path]
 */

const fs = require('fs');
const os = require('os');
const path = require('path');
const { ensureInstallEnv } = require('../install-root');

ensureInstallEnv();

const DEFAULT_STALE_MINUTES = 10;
const TRUNC_TASK = 60;
const TRUNC_SNIPPET = 120;

function printHelp() {
  const script = path.basename(process.argv[1]);
  console.log(
    [
      `Usage: node ${script} --master <id> [options]`,
      '',
      'Options:',
      '  --master <id>           Required manager agent id (flag name stays --master)',
      '  --stale-minutes <n>     Minutes after which feedback is considered stale (default 10)',
      '  --json                  Output JSON instead of a table',
      '  --limit <n>             Limit number of children shown',
      '  --out <path>            Write JSON aggregation to the given path',
      '  --help                  Show this help',
      '',
      'Examples:',
      `  node ${script} --master manager-123`,
      `  node ${script} --master manager-123 --stale-minutes 20 --json`,
      `  node ${script} --master manager-123 --out status.json`,
    ].join('\n'),
  );
}

function parseArgs(argv) {
  const opts = {
    masterId: '',
    staleMinutes: DEFAULT_STALE_MINUTES,
    json: false,
    limit: Infinity,
    outPath: '',
    help: false,
  };

  for (let i = 2; i < argv.length; i += 1) {
    const arg = argv[i];
    if (arg === '--master' || arg === '-m') {
      opts.masterId = argv[i + 1] || '';
      i += 1;
      continue;
    }
    if (arg === '--stale-minutes') {
      const raw = argv[i + 1];
      const num = Number(raw);
      if (!Number.isFinite(num) || num <= 0) {
        throw new Error('Invalid value for --stale-minutes; must be a positive number');
      }
      opts.staleMinutes = num;
      i += 1;
      continue;
    }
    if (arg === '--json') {
      opts.json = true;
      continue;
    }
    if (arg === '--limit') {
      const raw = argv[i + 1];
      const num = Number(raw);
      if (!Number.isFinite(num) || num <= 0) {
        throw new Error('Invalid value for --limit; must be a positive number');
      }
      opts.limit = num;
      i += 1;
      continue;
    }
    if (arg === '--out') {
      opts.outPath = argv[i + 1] || '';
      i += 1;
      continue;
    }
    if (arg === '--help' || arg === '-h') {
      opts.help = true;
      continue;
    }
    throw new Error(`Unknown argument: ${arg}`);
  }

  return opts;
}

function truncate(str, max) {
  if (!str) return '';
  if (str.length <= max) return str;
  return `${str.slice(0, max - 3)}...`;
}

function safeStat(filePath) {
  try {
    return fs.statSync(filePath);
  } catch (err) {
    return null;
  }
}

function splitBlocks(text) {
  const lines = text.split(/\r?\n/);
  const blocks = [];
  let current = [];
  lines.forEach((line) => {
    if (line.trim() === '') {
      if (current.length) {
        blocks.push(current);
        current = [];
      }
    } else {
      current.push(line);
    }
  });
  if (current.length) {
    blocks.push(current);
  }
  return blocks;
}

function readPromptSummary(promptPath) {
  const stat = safeStat(promptPath);
  if (!stat) {
    return { task: '', stat: null };
  }
  let content = '';
  try {
    content = fs.readFileSync(promptPath, 'utf8');
  } catch (err) {
    return { task: '', stat };
  }
  const lines = content.split(/\r?\n/);
  let summary = '';
  for (let i = 0; i < lines.length; i += 1) {
    const line = lines[i].trim();
    if (/^prompt\s*:/i.test(line)) {
      const inline = line.split(':').slice(1).join(':').trim();
      if (inline) {
        summary = inline;
        break;
      }
      const next = (lines[i + 1] || '').trim();
      if (next) {
        summary = next;
        break;
      }
    }
  }
  if (!summary) {
    const first = lines.find((l) => l.trim());
    summary = first ? first.trim() : '';
  }
  return { task: summary, stat };
}

function parseFeedback(feedbackPath, now) {
  const stat = safeStat(feedbackPath);
  if (!stat) {
    return { hasFeedback: false, stat: null, lastFeedbackAt: null, ageMinutes: null, snippet: '' };
  }
  let content = '';
  try {
    content = fs.readFileSync(feedbackPath, 'utf8');
  } catch (err) {
    return { hasFeedback: false, stat, lastFeedbackAt: null, ageMinutes: null, snippet: '' };
  }
  const blocks = splitBlocks(content);
  const latest = blocks.length ? blocks[blocks.length - 1] : null;
  let timestamp = null;
  let snippetLines = [];
  if (latest && latest.length) {
    const candidate = latest[0].trim();
    const parsed = new Date(candidate);
    if (!Number.isNaN(parsed.getTime())) {
      timestamp = parsed;
    }
    snippetLines = latest.slice(1).map((l) => l.trim()).filter(Boolean);
  }
  if (!timestamp) {
    timestamp = stat.mtime;
  }
  const lastFeedbackAt = timestamp ? timestamp.toISOString() : null;
  const ageMinutes = timestamp ? (now - timestamp) / 60000 : null;
  const snippet = snippetLines.slice(-3).join(' ');
  return {
    hasFeedback: true,
    stat,
    lastFeedbackAt,
    ageMinutes,
    snippet,
  };
}

function formatAge(minutes) {
  if (minutes === null || minutes === undefined) return '—';
  if (minutes < 1) {
    const secs = Math.max(Math.round(minutes * 60), 0);
    return `${secs}s`;
  }
  if (minutes < 90) return `${Math.round(minutes)}m`;
  const hours = minutes / 60;
  if (hours < 48) return `${Math.round(hours)}h`;
  const days = hours / 24;
  return `${Math.round(days)}d`;
}

function buildRows(masterDir, staleMinutes, limit, now) {
  const files = fs.readdirSync(masterDir);
  const ids = new Set();
  files.forEach((name) => {
    if (/^sub-.*-feedback\.txt$/i.test(name)) {
      ids.add(name.replace(/-feedback\.txt$/i, ''));
    } else if (/^sub-.*\.txt$/i.test(name)) {
      ids.add(name.replace(/\.txt$/i, ''));
    }
  });

  const rows = Array.from(ids)
    .sort()
    .slice(0, limit)
    .map((childId) => {
      const promptPath = path.join(masterDir, `${childId}.txt`);
      const feedbackPath = path.join(masterDir, `${childId}-feedback.txt`);
      const prompt = readPromptSummary(promptPath);
      const feedback = parseFeedback(feedbackPath, now);

      const ageMinutes = feedback.ageMinutes !== null ? Number(feedback.ageMinutes.toFixed(1)) : null;
      let status = 'no-feedback';
      if (feedback.hasFeedback) {
        if (ageMinutes !== null && ageMinutes > staleMinutes) {
          status = 'stale';
        } else {
          status = 'fresh';
        }
      }

      return {
        child_id: childId,
        status,
        task: prompt.task || '',
        last_feedback_at: feedback.lastFeedbackAt,
        age_minutes: ageMinutes,
        snippet: feedback.snippet || '',
        files: {
          prompt: promptPath,
          feedback: feedbackPath,
        },
        mtime: {
          prompt: prompt.stat ? prompt.stat.mtime.toISOString() : null,
          feedback: feedback.stat ? feedback.stat.mtime.toISOString() : null,
        },
      };
    });

  return rows;
}

function printTable(rows, staleMinutes) {
  if (!rows.length) {
    console.log('No agents found for this master.');
    return;
  }
  const headers = ['Child ID', 'Status', 'Age', 'Task', 'Snippet'];
  const widths = [20, 12, 8, TRUNC_TASK + 3, TRUNC_SNIPPET + 3];
  const formatRow = (cols) =>
    cols
      .map((col, idx) => {
        const str = col || '';
        return str.length > widths[idx] ? str.slice(0, widths[idx] - 3) + '...' : str.padEnd(widths[idx], ' ');
      })
      .join(' ');

  console.log(formatRow(headers));
  console.log(formatRow(headers.map((h) => '-'.repeat(h.length))));
  rows.forEach((row) => {
    const ageStr = row.status === 'no-feedback' ? '—' : formatAge(row.age_minutes);
    console.log(
      formatRow([
        row.child_id,
        row.status,
        ageStr,
        truncate(row.task.replace(/\s+/g, ' '), TRUNC_TASK),
        truncate(row.snippet.replace(/\s+/g, ' '), TRUNC_SNIPPET),
      ]),
    );
  });
  console.log(`Stale threshold: ${staleMinutes} minutes`);
}

function writeOut(pathToWrite, data) {
  try {
    fs.writeFileSync(pathToWrite, JSON.stringify(data, null, 2), 'utf8');
    console.error(`Wrote JSON to ${pathToWrite}`);
  } catch (err) {
    console.error(`Failed to write ${pathToWrite}: ${err.message}`);
    process.exitCode = 1;
  }
}

function main() {
  let opts;
  try {
    opts = parseArgs(process.argv);
  } catch (err) {
    console.error(err.message);
    printHelp();
    process.exit(1);
  }

  if (opts.help) {
    printHelp();
    process.exit(0);
  }

  if (!opts.masterId) {
    console.error('Missing required --master <id>');
    printHelp();
    process.exit(1);
  }

  const masterDir = path.join(os.tmpdir(), 'agentgarden', opts.masterId);
  if (!fs.existsSync(masterDir) || !fs.statSync(masterDir).isDirectory()) {
    console.error(`Manager temp dir not found: ${masterDir}`);
    process.exit(1);
  }

  const now = new Date();
  const rows = buildRows(masterDir, opts.staleMinutes, opts.limit, now);

  if (opts.json) {
    console.log(JSON.stringify(rows, null, 2));
  } else {
    printTable(rows, opts.staleMinutes);
  }

  if (opts.outPath) {
    writeOut(opts.outPath, rows);
  }
}

main();
