#!/usr/bin/env node
/**
 * Send a sub-agent spawn request to the running Electron app.
 *
 * Usage: node spawn_subagent.js <master_agent_id> "<prompt>"
 */

const net = require('net');
const crypto = require('crypto');
const { ensureInstallEnv, pathFromInstall } = require('../install-root');

const PORT = parseInt(process.env.SUBAGENT_PORT || '32123', 10);
const HOST = '127.0.0.1';
ensureInstallEnv();

function main() {
  const [master_agent_id, prompt] = process.argv.slice(2);
  if (!master_agent_id || !prompt) {
    console.error('Usage: node spawn_subagent.js <master_agent_id> "<prompt>"');
    process.exit(1);
  }

  const child_id = `sub-${Date.now()}-${crypto.randomBytes(3).toString('hex')}`;
  const feedbackScript = pathFromInstall('scripts', 'agents', 'submit_feedback.js');
  const augmentedPrompt = `${prompt}\n\nYou are a sub agent codex terminal that was spawned by a master agent in a framework designed to enable hierarchical agent reasoning. You are an engineer (not just a scout) and can make code changes. Only proceed with edits when the master has provided extremely detailed instructions (files, steps, constraints); otherwise ask for the missing detail first.\n\nWhen you reach a conclusion, proactively report back without being asked by running:\nnode ${feedbackScript} ${master_agent_id} ${child_id} "<detailed feedback>"\n\nBe exhaustive and descriptive in that feedback: include what you tried, what worked, what failed, any assumptions, any edits you made, and follow-ups you recommend.  Also, you should not wait until your task is fully finished to provide feedback to the master agent.  Instead, continuously provide feedback to assure the master agent you are thinking and making progress.  Send an initial response when you first start working on the task. then send periodic updates as you make progress. then send a final response when you are done.\n`;

  const payload = JSON.stringify({ master_agent_id, prompt: augmentedPrompt, child_id }) + '\n';
  const socket = net.createConnection({ host: HOST, port: PORT }, () => {
    socket.write(payload);
    socket.end();
  });

  socket.on('error', (err) => {
    console.error('Failed to send sub-agent request:', err.message);
    process.exit(1);
  });
}

main();
