#!/usr/bin/env node
/**
 * Append feedback from a sub-agent to the master's temp folder.
 * Usage: node submit_feedback.js <master_id> <child_id> "<feedback>"
 */

const fs = require('fs');
const os = require('os');
const path = require('path');
const net = require('net');
const { ensureInstallEnv, pathFromInstall } = require('../install-root');

ensureInstallEnv();

function ensureDir(dir) {
  if (!fs.existsSync(dir)) {
    fs.mkdirSync(dir, { recursive: true });
  }
}

function writeFile(masterId, childId, feedback) {
  const root = path.join(os.tmpdir(), 'agentgarden');
  const masterDir = path.join(root, masterId);
  ensureDir(masterDir);
  const filePath = path.join(masterDir, `${childId}-feedback.txt`);
  const entry = `${new Date().toISOString()}\n${feedback}\n\n`;
  fs.appendFileSync(filePath, entry, 'utf8');
  return filePath;
}

function notifyGui(masterId, childId, feedback, direction) {
  let prompt = feedback;
  if (direction === 'to-child') {
    const scriptSelf = pathFromInstall('scripts', 'agents', 'submit_feedback.js');
    const feedbackFile = path.join(os.tmpdir(), 'agentgarden', masterId, `${childId}-feedback.txt`);
    prompt = `${feedback}\n\nWhen you reply, log detailed feedback by running:\nnode ${scriptSelf} ${masterId} ${childId} "<feedback>"\n(This writes to ${feedbackFile})`;
  }

  const payload = JSON.stringify({
    kind: 'feedback',
    master_agent_id: masterId,
    child_id: childId,
    prompt,
    direction,
  }) + '\n';
  const port = parseInt(process.env.SUBAGENT_PORT || '32123', 10);
  const socket = net.createConnection({ host: '127.0.0.1', port }, () => {
    socket.write(payload);
    socket.end();
  });
  socket.on('error', () => {}); // silently ignore if GUI not running
}

function main() {
  const args = process.argv.slice(2);
  const toChildIdx = args.indexOf('--to-child');
  const toChild = toChildIdx !== -1;
  if (toChildIdx !== -1) {
    args.splice(toChildIdx, 1);
  }

  const [masterId, childId, ...rest] = args;
  const feedback = (rest || []).join(' ').trim();
  if (!masterId || !childId || !feedback) {
    console.error('Usage: node submit_feedback.js <master_id> <child_id> "<feedback>" [--to-child]');
    process.exit(1);
  }

  const filePath = writeFile(masterId, childId, feedback);
  notifyGui(masterId, childId, feedback, toChild ? 'to-child' : 'to-master');
  console.log(`Feedback written to ${filePath}`);
}

main();
