#!/usr/bin/env node
/**
 * Update the master agent title (plus optional body and final summary) in the running GUI.
 * Usage: node update_title.js <master_id> "<short title>" [--body "<working body>"] ["<100-word summary>" | --summary "<100-word summary>"]
 */

const net = require('net');
const { ensureInstallEnv } = require('../install-root');

const HOST = '127.0.0.1';
const PORT = parseInt(process.env.SUBAGENT_PORT || '32123', 10);
ensureInstallEnv();

function main() {
  const args = process.argv.slice(2);
  const masterId = (args.shift() || '').trim();
  const title = (args.shift() || '').trim();
  let body = '';
  let summary = '';

  for (let i = 0; i < args.length; i += 1) {
    const token = args[i];
    if (token === '--body') {
      body = (args[i + 1] || '').trim();
      i += 1;
      continue;
    }
    if (token === '--summary') {
      summary = (args[i + 1] || '').trim();
      i += 1;
      continue;
    }
    if (!summary) {
      summary = token;
    } else {
      summary += ` ${token}`;
    }
  }

  if (!masterId || !title) {
    console.error(
      'Usage: node update_title.js <master_id> "<short title>" [--body "<working body>"] ["<100-word summary>" | --summary "<100-word summary>"]',
    );
    process.exit(1);
  }

  if (body) {
    const words = body.split(/\s+/).filter(Boolean);
    if (words.length > 200) {
      console.error(`Body is too long: ${words.length} words (max 200).`);
      process.exit(1);
    }
  }

  if (summary) {
    const words = summary.split(/\s+/).filter(Boolean);
    if (words.length > 100) {
      console.error(`Summary is too long: ${words.length} words (max 100).`);
      process.exit(1);
    }
  }

  const payload = JSON.stringify({
    kind: 'title',
    master_agent_id: masterId,
    title,
    body: body || undefined,
    summary: summary || undefined,
  }) + '\n';

  const socket = net.createConnection({ host: HOST, port: PORT }, () => {
    socket.write(payload);
    socket.end();
  });

  socket.on('error', (err) => {
    console.error(`Failed to send title update: ${err.message}`);
    process.exit(1);
  });
}

main();
