const { app, BrowserWindow, ipcMain, dialog, shell } = require('electron');
const net = require('net');
const path = require('path');
const pty = require('node-pty');
const fs = require('fs');
const os = require('os');
const { ensureInstallEnv, pathFromInstall } = require('../install-root');

let mainWindow;
const terminals = new Map();
const SUBAGENT_PORT = parseInt(process.env.SUBAGENT_PORT || '32123', 10);
const BRING_TO_FRONT =
  process.env.AGENTGARDEN_BRING_TO_FRONT === '1' || process.env.HAM_BRING_TO_FRONT === '1';
let subagentServerStarted = false;
const LOG_ROOT = path.join(os.tmpdir(), 'agentgarden');
let cleanupInProgress = false;
ensureInstallEnv();

function cleanTempRoot() {
  try {
    if (fs.existsSync(LOG_ROOT)) {
      fs.rmSync(LOG_ROOT, { recursive: true, force: true });
    }
  } catch (err) {
    console.error('Failed to clean temp logs:', err);
  }
}

function killAllTerminals() {
  terminals.forEach((term) => term.kill());
  terminals.clear();
}

function cleanupAll({ exit = false, exitCode = 0 } = {}) {
  if (cleanupInProgress) return;
  cleanupInProgress = true;
  try {
    killAllTerminals();
    cleanTempRoot();
  } catch (err) {
    console.error('Cleanup error:', err);
  } finally {
    if (exit) {
      try {
        app.quit();
      } catch (e) {
        // ignore
      }
      process.exit(exitCode);
    }
  }
}

// Enforce single instance so external spawns always hit the same window
const gotLock = app.requestSingleInstanceLock();
if (!gotLock) {
  app.quit();
  process.exit(0);
} else {
  // Best-effort removal of stale temp files from any previous crash/exit
  cleanTempRoot();
  app.on('second-instance', () => {
    if (mainWindow) {
      if (mainWindow.isMinimized()) mainWindow.restore();
      mainWindow.show();
      mainWindow.focus();
    }
  });
}

function createWindow() {
  mainWindow = new BrowserWindow({
    width: 1400,
    height: 900,
    backgroundColor: '#0f172a',
    titleBarStyle: 'hiddenInset',
    webPreferences: {
      contextIsolation: false,
      nodeIntegration: true,
    },
  });

  mainWindow.loadFile(pathFromInstall('src', 'index.html'));
  if (process.argv.includes('--dev')) {
    // DevTools start closed by default; open manually if needed.
  }
}

app.whenReady().then(createWindow);
app.whenReady().then(startSubagentServer);

function deliverToRenderer(channel, payload) {
  if (!mainWindow || mainWindow.isDestroyed()) return;
  if (BRING_TO_FRONT) {
    if (mainWindow.isMinimized()) mainWindow.restore();
    mainWindow.show();
    mainWindow.focus();
  }
  mainWindow.webContents.send(channel, payload);
}

app.on('window-all-closed', () => {
  cleanupAll();
  if (process.platform !== 'darwin') {
    app.quit();
  }
});

process.on('exit', () => {
  cleanupAll();
});

['SIGINT', 'SIGTERM', 'SIGHUP'].forEach((sig) => {
  process.on(sig, () => {
    cleanupAll({ exit: true, exitCode: 0 });
  });
});

process.on('uncaughtException', (err) => {
  console.error('Uncaught exception:', err);
  cleanupAll({ exit: true, exitCode: 1 });
});

process.on('unhandledRejection', (reason) => {
  console.error('Unhandled rejection:', reason);
  cleanupAll({ exit: true, exitCode: 1 });
});

function startSubagentServer() {
  if (subagentServerStarted) return;
  subagentServerStarted = true;
  const server = net.createServer((socket) => {
    let buffer = '';
    socket.on('data', (chunk) => {
      buffer += chunk.toString();
      let idx;
      while ((idx = buffer.indexOf('\n')) >= 0) {
        const line = buffer.slice(0, idx).trim();
        buffer = buffer.slice(idx + 1);
        if (!line) continue;
        try {
          const payload = JSON.parse(line);
          if (payload.kind === 'feedback') {
            if (mainWindow && !mainWindow.isDestroyed() && payload.child_id && payload.prompt) {
              deliverToRenderer('external-feedback', payload);
            }
          } else if (payload.kind === 'kill') {
            if (mainWindow && !mainWindow.isDestroyed() && payload.child_id) {
              deliverToRenderer('external-kill', payload);
            }
          } else if (payload.kind === 'title') {
            const hasTitleOrSummary = Boolean(payload.title) || typeof payload.summary === 'string';
            if (mainWindow && !mainWindow.isDestroyed() && payload.master_agent_id && hasTitleOrSummary) {
              deliverToRenderer('external-title', payload);
            }
          } else if (payload.master_agent_id && payload.prompt) {
            if (mainWindow && !mainWindow.isDestroyed()) {
              deliverToRenderer('external-subagent', payload);
            }
          }
        } catch (err) {
          // ignore malformed lines
        }
      }
    });
  });

  server.on('error', (err) => {
    if (err.code === 'EADDRINUSE') {
      console.error(`Subagent server port ${SUBAGENT_PORT} already in use; skipping listener.`);
      if (mainWindow && !mainWindow.isDestroyed()) {
        deliverToRenderer('subagent-server-error', {
          message: `Port ${SUBAGENT_PORT} already in use; external spawn disabled.`,
        });
      }
      return;
    }
    console.error('Subagent server error:', err);
  });

  server.listen(SUBAGENT_PORT, '127.0.0.1', () => {
    console.log(`Subagent server listening on 127.0.0.1:${SUBAGENT_PORT}`);
  });
}

app.on('activate', () => {
  if (BrowserWindow.getAllWindows().length === 0) {
    createWindow();
  }
});

// IPC handlers
ipcMain.handle('create-terminal', (event, { id, cwd, command, args, env }) => {
  try {
    const shell = command || process.env.SHELL || '/bin/bash';
    const spawnArgs = Array.isArray(args) && args.length ? args : ['-l'];

    const ptyProcess = pty.spawn(shell, spawnArgs, {
      name: 'xterm-256color',
      cols: 100,
      rows: 30,
      cwd: cwd || process.cwd(),
      env: {
        ...process.env,
        ...(env || {}),
        TERM: 'xterm-256color',
        TERM_PROGRAM: '',
      },
    });

    terminals.set(id, ptyProcess);

    ptyProcess.onData((data) => {
      mainWindow.webContents.send('terminal-data', { id, data });
    });

    ptyProcess.onExit(({ exitCode }) => {
      mainWindow.webContents.send('terminal-exit', { id, exitCode });
      terminals.delete(id);
    });

    return { success: true };
  } catch (error) {
    return { success: false, error: error.message };
  }
});

ipcMain.handle('terminal-input', (event, { id, data }) => {
  const term = terminals.get(id);
  if (term) {
    term.write(data);
    return { success: true };
  }
  return { success: false, error: 'terminal not found' };
});

ipcMain.handle('terminal-resize', (event, { id, cols, rows }) => {
  const term = terminals.get(id);
  if (term) {
    term.resize(cols, rows);
    return { success: true };
  }
  return { success: false, error: 'terminal not found' };
});

ipcMain.handle('close-terminal', (event, { id }) => {
  const term = terminals.get(id);
  if (term) {
    term.kill();
    terminals.delete(id);
    return { success: true };
  }
  return { success: false, error: 'terminal not found' };
});

ipcMain.handle('select-directory', async () => {
  const win = BrowserWindow.getFocusedWindow() || mainWindow;
  const result = await dialog.showOpenDialog(win, {
    properties: ['openDirectory', 'createDirectory'],
  });
  return {
    canceled: result.canceled,
    path: Array.isArray(result.filePaths) && result.filePaths.length ? result.filePaths[0] : null,
  };
});

ipcMain.handle('record-task', async (event, { masterId, childId, prompt }) => {
  try {
    if (!masterId || !childId) {
      return { success: false, error: 'missing masterId or childId' };
    }
    if (!fs.existsSync(LOG_ROOT)) {
      fs.mkdirSync(LOG_ROOT, { recursive: true });
    }
    const masterDir = path.join(LOG_ROOT, masterId);
    if (!fs.existsSync(masterDir)) {
      fs.mkdirSync(masterDir, { recursive: true });
    }
    const filePath = path.join(masterDir, `${childId}.txt`);
    const content = [`Prompt: ${prompt || ''}`, `Timestamp: ${new Date().toISOString()}`].join('\n');
    fs.writeFileSync(filePath, content, 'utf8');
    return { success: true, filePath };
  } catch (err) {
    return { success: false, error: err.message };
  }
});

ipcMain.handle('delete-master-temp', async (event, { masterId }) => {
  try {
    if (!masterId) return { success: false, error: 'missing masterId' };
    const targetDir = path.join(LOG_ROOT, masterId);
    if (!targetDir.startsWith(LOG_ROOT)) {
      return { success: false, error: 'invalid path' };
    }
    if (fs.existsSync(targetDir)) {
      fs.rmSync(targetDir, { recursive: true, force: true });
    }
    return { success: true };
  } catch (err) {
    return { success: false, error: err.message };
  }
});

ipcMain.handle('open-feedback-file', async (event, { masterId, childId }) => {
  try {
    if (!masterId || !childId) return { success: false, error: 'missing masterId or childId' };
    const masterDir = path.join(LOG_ROOT, masterId);
    if (!masterDir.startsWith(LOG_ROOT)) return { success: false, error: 'invalid path' };
    if (!fs.existsSync(masterDir)) {
      fs.mkdirSync(masterDir, { recursive: true });
    }
    const feedbackPath = path.join(masterDir, `${childId}-feedback.txt`);
    const altPath = path.join(masterDir, `${childId}.txt`);
    const target = fs.existsSync(feedbackPath) ? feedbackPath : fs.existsSync(altPath) ? altPath : feedbackPath;
    if (!fs.existsSync(target)) {
      fs.writeFileSync(target, '', 'utf8');
    }
    const res = await shell.openPath(target);
    if (res) {
      return { success: false, error: res };
    }
    return { success: true, path: target };
  } catch (err) {
    return { success: false, error: err.message };
  }
});
