const { Terminal } = require('@xterm/xterm');
const { FitAddon } = require('@xterm/addon-fit');
const { ipcRenderer } = require('electron');
const { spawnSync } = require('child_process');
const path = require('path');
const fs = require('fs');
const os = require('os');
const { ensureInstallEnv, pathFromInstall } = require('../install-root');

const { envVar: INSTALL_ENV_VAR } = ensureInstallEnv();
const AGENT_PATH_STORAGE_KEY = 'ham_agent_paths';

const terminalAPI = (window && window.terminalAPI) || {
  create: (payload) => ipcRenderer.invoke('create-terminal', payload),
  send: (payload) => ipcRenderer.invoke('terminal-input', payload),
  resize: (payload) => ipcRenderer.invoke('terminal-resize', payload),
  close: (payload) => ipcRenderer.invoke('close-terminal', payload),
  onData: (handler) => ipcRenderer.on('terminal-data', handler),
  onExit: (handler) => ipcRenderer.on('terminal-exit', handler),
};

const canvas = document.getElementById('canvas');
const controlPanel = document.getElementById('control-panel');
const sidebarToggle = document.getElementById('sidebar-toggle');
const centerMasterBtn = document.getElementById('center-master');
const world = document.createElement('div');
world.id = 'world';
canvas.insertBefore(world, canvas.firstChild);
const promptText = document.getElementById('prompt-text');
const copyPromptBtn = document.getElementById('copy-prompt');
const terminalDepot = document.getElementById('terminal-depot');
const canvasBody = document.getElementById('canvas');
const chooseFolderBtn = document.getElementById('choose-folder');
const folderPathEl = document.getElementById('folder-path');
const settingsButtons = Array.from(document.querySelectorAll('.settings-btn'));
const settingsOverlay = document.getElementById('settings-overlay');
const settingsCloseBtn = document.getElementById('settings-close');
const masterListEl = document.getElementById('master-list');
const masterEmptyEl = document.getElementById('master-empty');
const childListEl = document.getElementById('child-list');
const childEmptyEl = document.getElementById('child-empty');
const childSectionTitle = document.querySelector('#child-section .section-title');
const childSectionEl = document.getElementById('child-section');
const workingDirHelpBtn = document.getElementById('working-dir-help');
const workingDirTooltip = document.getElementById('working-dir-tooltip');
const agentStatusEls = Array.from(document.querySelectorAll('[data-agent-status]'));
const agentButtons = Array.from(document.querySelectorAll('[data-agent-option]'));
const agentDefaultInputs = Array.from(document.querySelectorAll('input[data-agent-default]'));
const agentHelpBtn = document.getElementById('agent-help');
const agentTooltip = document.getElementById('agent-tooltip');
const agentPathInputs = new Map(); // id -> array of inputs
document.querySelectorAll('[data-agent-path]').forEach((input) => {
  const id = input.dataset.agentPath;
  if (!id) return;
  if (!agentPathInputs.has(id)) agentPathInputs.set(id, []);
  agentPathInputs.get(id).push(input);
});
agentDefaultInputs.forEach((input) => {
  input.addEventListener('change', () => {
    if (input.checked && input.value) {
      setSelectedAgent(input.value);
    }
  });
});
const childPanels = new Map();
const childNames = new Map();
const familyAssignments = new Map();
const usedFamilies = new Set();
const usedFlowers = new Set();
const FAMILY_SETS = [
  {
    family: 'asteraceae',
    flowers: [
      'daisy', 'sunflower', 'marigold', 'zinnia', 'dahlia', 'chrysanthemum', 'aster', 'tansy', 'yarrow',
      'coneflower', 'gaillardia', 'calendula', 'cosmos', 'tithonia', 'blackeyedsusan',
    ],
  },
  {
    family: 'rosaceae',
    flowers: [
      'rose', 'hawthorn', 'quince', 'crabapple', 'pearblossom', 'cherryblossom', 'peachblossom',
      'apricotblossom', 'almondblossom', 'spirea', 'meadowsweet', 'ladybanks', 'eglantine', 'briar',
      'burnet',
    ],
  },
  {
    family: 'liliaceae',
    flowers: [
      'lily', 'tulip', 'fritillaria', 'hyacinth', 'daylily', 'trillium', 'erythronium', 'camas', 'asphodel',
      'colchicum', 'bluebell', 'allium',
    ],
  },
  {
    family: 'orchidaceae',
    flowers: [
      'orchid', 'cattleya', 'cymbidium', 'phalaenopsis', 'dendrobium', 'oncidium', 'paphiopedilum',
      'vanilla', 'vanda', 'miltonia', 'brassia', 'epidendrum', 'laelia', 'coelogyne',
    ],
  },
  {
    family: 'iridaceae',
    flowers: [
      'iris', 'crocus', 'freesia', 'gladiolus', 'ixia', 'sparaxis', 'tigridia', 'watsonia', 'sisyrinchium',
      'babiana', 'libertia', 'moraea',
    ],
  },
  {
    family: 'fabaceae',
    flowers: [
      'sweetpea', 'lupine', 'wisteria', 'clover', 'mimosa', 'indigofera', 'locust', 'gorse', 'laburnum',
      'broom', 'redbud', 'sophora', 'alfalfa', 'pea',
    ],
  },
  {
    family: 'amaryllidaceae',
    flowers: [
      'amaryllis', 'daffodil', 'snowdrop', 'nerine', 'agapanthus', 'clivia', 'crinum', 'leucojum',
      'hippeastrum', 'habranthus', 'lycoris',
    ],
  },
  {
    family: 'ranunculaceae',
    flowers: [
      'anemone', 'clematis', 'buttercup', 'delphinium', 'larkspur', 'columbine', 'hellebore', 'peony',
      'aconite', 'pasqueflower', 'trollius',
    ],
  },
  {
    family: 'campanulaceae',
    flowers: [
      'bellflower', 'campanula', 'lobelia', 'platycodon', 'balloonflower', 'canterburybells', 'adenophora',
      'githopsis',
    ],
  },
  {
    family: 'caryophyllaceae',
    flowers: [
      'carnation', 'gypsophila', 'dianthus', 'silene', 'soapwort', 'sandwort', 'campion', 'babybreath',
    ],
  },
];
const shuffledFamilies = shuffleArray([...FAMILY_SETS]);
let familyCursor = 0;
const FALLBACK_FLOWERS = FAMILY_SETS.flatMap((f) => f.flowers);
let flowerIndex = 0;
const feedbackLogs = new Map();
const feedbackMarkers = new Map();

let terminalCount = 0;
const terminals = new Map();
const masterSlots = new Map();
const childToSlot = new Map();
const masterTitles = new Map();
const pendingTitles = new Map();
const masterSummaries = new Map();
const pendingSummaries = new Map();
const pendingBodies = new Map();
const masterPanels = new Map();
const THINKING_IDLE_MS = 2400;
const viewState = { scale: 1, tx: 0, ty: 0 };
const MIN_SCALE = 0.4;
const MAX_SCALE = 1.8;
const FIT_ANIM_DURATION_MS = 340;
const canvasCenterCache = {
  rect: null,
  local: { x: 0, y: 0 },
  world: { x: 0, y: 0 },
};
let panAnimation = null;
let isPanning = false;
let panEngaged = false;
let panStart = { x: 0, y: 0 };
let panOrigin = { tx: 0, ty: 0 };
let sidebarCollapsed = true;
let recenterTimer = null;
const MASTER_BUFFER = 40;
const DEFAULT_SHELL_SIZE = 300;
const DEFAULT_SLOT_SIZE = 180;
const DEFAULT_ORBIT = 200;
const AGENT_STORAGE_KEY = 'ham_agent_runtime';
const AGENT_OPTIONS = [
  { id: 'codex', label: 'Codex', binary: 'codex' },
  { id: 'claude', label: 'Claude', binary: 'claude' },
  { id: 'gemini', label: 'Gemini', binary: 'gemini' },
];
let agentAvailability = {};
let agentResolvedPaths = {};
let agentPathOverrides = {};
agentPathOverrides = loadAgentPathOverrides();
syncAgentPathInputs();
agentPathInputs.forEach((inputs, agentId) => {
  inputs.forEach((input) => {
    const handler = () => {
      const value = (input.value || '').trim();
      if (value) {
        agentPathOverrides[agentId] = value;
      } else {
        delete agentPathOverrides[agentId];
      }
      persistAgentPathOverrides();
      syncAgentPathInputs();
      refreshAgentDetection();
    };
    input.addEventListener('change', handler);
    input.addEventListener('blur', handler);
  });
});
let selectedAgent = AGENT_OPTIONS[0].id;
const PLACEHOLDER_MASTER_ID = 'placeholder-master';
let activeTerminalId = null;
let activeChildPanelId = null;
let selectedCwd = process.cwd ? process.cwd() : '';
let lastMasterId = null;

if (sidebarCollapsed) {
  document.body.classList.add('sidebar-collapsed');
}

const MASTER_PROMPT_TEMPLATE_PATH = pathFromInstall('master_helper_prompt.txt');
const DEFAULT_MASTER_PROMPT_TEMPLATE = [
  'Codex terminals will start in: {{WORKING_DIRECTORY}}',
  'Your job as manager: plan, orchestrate, delegate. Do NOT do the nitty gritty work yourself - spawn sub-agents for that.',
  'Sub-agents are engineers, not just scouts. When asking them to edit code, give them extreme detail (files, steps, constraints) before they touch anything.',
  'Expect extreme detail back from sub-agents: what they tried, what changed, what failed, and follow-ups needed.',
  'Current manager id: {{MASTER_ID}}',
  'Reuse this id for every sub-agent spawn, feedback, and title update.',
  'Keep the manager title fresh (8 words or fewer) by running:',
  'node {{TITLE_SCRIPT}} {{MASTER_ID}} "<short manager title>"',
  'Add a short working body under the title by including --body "<current focus>"',
  'Update it periodically so it reflects the current focus.',
  'When the entire request is done, send the final <=100 word recap (shows inside the manager circle) alongside the title:',
  'node {{TITLE_SCRIPT}} {{MASTER_ID}} "<short manager title>" "<100-word final summary>"',
  'You can spawn sub-agents using the local script.',
  'When you have a subtask, run:',
  'node {{SPAWN_SCRIPT}} {{MASTER_ID}} "<subtask prompt>"',
  'Always use the exact manager id shown above.',
  'Each child writes feedback back to you by calling:',
  'node {{FEEDBACK_SCRIPT}} {{MASTER_ID}} <child_id> "<feedback>"',
  'Find child feedback in:',
  '{{FEEDBACK_ROOT}}/{{MASTER_ID}}/<child_id>-feedback.txt',
  'Keep the manager terminal open and monitor feedback files for child results.',
  'To check sub-agent progress from the manager terminal, run:',
  'node {{STATUS_SCRIPT}} --master {{MASTER_ID}}',
  'To send a follow-up to a child via script, use:',
  'node {{FEEDBACK_SCRIPT}} {{MASTER_ID}} <child_id> "<feedback>" --to-child',
  "When you are satisfied with a child's result, close its terminal by running:",
  'node {{KILL_SCRIPT}} {{MASTER_ID}} <child_id>',
  'Click any manager or child circle to select it; its terminal shows inside the matching panel on the left.',
  'Use the expanded panel to type into whichever terminal is selected.',
  'If SUBAGENT_PORT is set, the script and app must share it.',
  'Sample (using the id above): Spawn two sub-agents to each tell me a random color. Once you see their responses, respond back to them each asking for a different color, then close each child with the kill script when done.',
].join('\n');
let cachedMasterPromptTemplate = null;
let cachedMasterPromptMtime = null;
let masterPromptErrorLogged = false;

function applyTransform() {
  world.style.transform = `translate(${viewState.tx}px, ${viewState.ty}px) scale(${viewState.scale})`;
  updateCanvasCenterCache();
}

function scheduleRecenterOnLayout() {
  if (recenterTimer) cancelAnimationFrame(recenterTimer);
  recenterTimer = requestAnimationFrame(() => {
    recenterTimer = requestAnimationFrame(() => {
      recenterTimer = null;
      updateCanvasCenterCache();
    });
  });
}

function screenToWorld(clientX, clientY) {
  const rect = canvas.getBoundingClientRect();
  return {
    x: (clientX - rect.left - viewState.tx) / viewState.scale,
    y: (clientY - rect.top - viewState.ty) / viewState.scale,
  };
}

function canvasCenterWorld() {
  if (!canvasCenterCache.rect) {
    updateCanvasCenterCache();
  }
  return canvasCenterCache.world;
}

function updateCanvasCenterCache() {
  if (!canvas) return canvasCenterCache;
  const rect = canvas.getBoundingClientRect();
  const sidebarRect = controlPanel?.getBoundingClientRect();
  const leftInset = sidebarRect ? sidebarRect.width : 0;
  const local = {
    x: leftInset + (rect.width - leftInset) / 2,
    y: rect.height / 2,
  };
  const world = {
    x: (local.x - viewState.tx) / viewState.scale,
    y: (local.y - viewState.ty) / viewState.scale,
  };
  canvasCenterCache.rect = rect;
  canvasCenterCache.local = local;
  canvasCenterCache.world = world;
  return canvasCenterCache;
}

function clamp(value, min, max) {
  return Math.min(Math.max(value, min), max);
}

function shuffleArray(arr) {
  const a = [...arr];
  for (let i = a.length - 1; i > 0; i -= 1) {
    const j = Math.floor(Math.random() * (i + 1));
    [a[i], a[j]] = [a[j], a[i]];
  }
  return a;
}

function openFlowerSearch(term) {
  if (!term) return;
  const q = `${term} flower`;
  try {
    window.open(
      `https://www.google.com/search?tbm=isch&q=${encodeURIComponent(q)}`,
      '_blank',
      'noopener,noreferrer',
    );
  } catch (err) {
    console.error('Failed to open search', err);
  }
}

function attachSearchLink(el, term) {
  if (!el || !term) return;
  el.dataset.searchTerm = term;
  if (!el.dataset.searchBound) {
    el.addEventListener('click', (e) => {
      e.stopPropagation();
      openFlowerSearch(el.dataset.searchTerm || term);
    });
    el.dataset.searchBound = '1';
  }
}

function updateSidebarToggle() {
  if (!sidebarToggle) return;
  const label = sidebarCollapsed ? 'Expand control panel' : 'Collapse control panel';
  sidebarToggle.setAttribute('aria-label', label);
  sidebarToggle.title = label;
}

function offsetForSidebarShift(beforeRect) {
  if (!canvas || !beforeRect) return;
  requestAnimationFrame(() => {
    const after = canvas.getBoundingClientRect();
    if (!after) return;
    const deltaL = after.left - beforeRect.left;
    if (Math.abs(deltaL) < 0.5) return;
    animateViewport({ tx: viewState.tx - deltaL * 0.5, ty: viewState.ty, scale: viewState.scale });
    scheduleRecenterOnLayout();
  });
}

function expandSidebar() {
  const beforeRect = canvas?.getBoundingClientRect();
  if (!sidebarCollapsed) return;
  sidebarCollapsed = false;
  document.body.classList.remove('sidebar-collapsed');
  updateSidebarToggle();
  offsetForSidebarShift(beforeRect);
  updateCanvasCenterCache();
  scheduleRecenterOnLayout();
}

function collapseSidebar({ skipOffset = false } = {}) {
  const beforeRect = canvas?.getBoundingClientRect();
  if (sidebarCollapsed) return;
  sidebarCollapsed = true;
  document.body.classList.add('sidebar-collapsed');
  collapseAllChildPanels();
  collapseAllMasterPanels();
  updateSidebarToggle();
  if (!skipOffset) {
    offsetForSidebarShift(beforeRect);
  }
  updateCanvasCenterCache();
  scheduleRecenterOnLayout();
}

function toggleSidebar() {
  if (sidebarCollapsed) {
    expandSidebar();
  } else {
    collapseSidebar();
  }
}

function stopPanAnimation() {
  if (!panAnimation) return;
  panAnimation.stopped = true;
  if (panAnimation.rafId) {
    cancelAnimationFrame(panAnimation.rafId);
  }
  panAnimation = null;
}

function panEase(t) {
  // Accelerate into the motion, then ease out near the end.
  return t < 0.5 ? 4 * t * t * t : 1 - Math.pow(-2 * t + 2, 3) / 2;
}

function elementBounds(el) {
  if (!el) return null;
  const circle = circleInfo(el, 0);
  return {
    left: circle.x - circle.r,
    top: circle.y - circle.r,
    right: circle.x + circle.r,
    bottom: circle.y + circle.r,
  };
}

function fitViewportToBounds(bounds) {
  if (!bounds || !canvas) return null;
  const metrics = canvasCenterCache.rect ? canvasCenterCache : updateCanvasCenterCache();
  const rect = metrics?.rect || canvas.getBoundingClientRect();
  const targetScale = clamp(viewState.scale, MIN_SCALE, MAX_SCALE);
  const cx = (bounds.left + bounds.right) / 2;
  const cy = (bounds.top + bounds.bottom) / 2;
  const screenCenterX = metrics?.local?.x ?? rect.width / 2;
  const screenCenterY = metrics?.local?.y ?? rect.height / 2;
  const tx = screenCenterX - cx * targetScale;
  const ty = screenCenterY - cy * targetScale;
  return { tx, ty, scale: targetScale };
}

function animateViewport(target) {
  if (!target) return;
  const start = { tx: viewState.tx, ty: viewState.ty, scale: viewState.scale };
  const needsMove = Math.hypot(target.tx - start.tx, target.ty - start.ty) > 0.5;
  const needsZoom = Math.abs(target.scale - start.scale) > 0.0001;
  const duration = needsMove || needsZoom ? FIT_ANIM_DURATION_MS : 0;
  stopPanAnimation();
  if (duration === 0) {
    viewState.tx = target.tx;
    viewState.ty = target.ty;
    viewState.scale = clamp(target.scale, MIN_SCALE, MAX_SCALE);
    applyTransform();
    return;
  }
  const startTime = performance.now();
  const animation = { rafId: null, stopped: false };
  const step = () => {
    if (animation.stopped) return;
    const now = performance.now();
    const progress = clamp((now - startTime) / duration, 0, 1);
    const eased = panEase(progress);
    viewState.tx = start.tx + (target.tx - start.tx) * eased;
    viewState.ty = start.ty + (target.ty - start.ty) * eased;
    viewState.scale = start.scale + (target.scale - start.scale) * eased;
    applyTransform();
    if (progress < 1) {
      animation.rafId = requestAnimationFrame(step);
    } else {
      panAnimation = null;
    }
  };
  panAnimation = animation;
  animation.rafId = requestAnimationFrame(step);
}

function shuffleArray(arr) {
  const a = [...arr];
  for (let i = a.length - 1; i > 0; i -= 1) {
    const j = Math.floor(Math.random() * (i + 1));
    [a[i], a[j]] = [a[j], a[i]];
  }
  return a;
}

function generateFlowerId() {
  const start = flowerIndex;
  const total = FALLBACK_FLOWERS.length;
  for (let i = 0; i < total; i += 1) {
    const name = FALLBACK_FLOWERS[(start + i) % total];
    if (!terminals.has(name)) {
      flowerIndex = (start + i + 1) % total;
      return name;
    }
  }
  // Fallback with suffix if all are taken
  const fallback = FALLBACK_FLOWERS[flowerIndex % total];
  flowerIndex = (flowerIndex + 1) % total;
  return `${fallback}-${Date.now().toString(36)}`;
}

function pickFamilyKey() {
  for (let i = 0; i < shuffledFamilies.length; i += 1) {
    const idx = (familyCursor + i) % shuffledFamilies.length;
    const key = shuffledFamilies[idx].family;
    if (!usedFamilies.has(key)) {
      familyCursor = (idx + 1) % shuffledFamilies.length;
      return key;
    }
  }
  return null;
}

function ensureFamilyForMaster(masterId) {
  if (familyAssignments.has(masterId)) return familyAssignments.get(masterId);
  const key = pickFamilyKey();
  if (!key) return null;
  familyAssignments.set(masterId, key);
  usedFamilies.add(key);
  return key;
}

function releaseFamilyForMaster(masterId) {
  const key = familyAssignments.get(masterId);
  if (key) {
    familyAssignments.delete(masterId);
    usedFamilies.delete(key);
  }
}

function familyConfigForMaster(masterId) {
  const key = familyAssignments.get(masterId);
  if (!key) return null;
  return FAMILY_SETS.find((f) => f.family === key) || null;
}

function allocateFlowerForMaster(masterId) {
  const cfg = familyConfigForMaster(masterId);
  const pool = cfg?.flowers || FLOWER_NAMES;
  for (let i = 0; i < pool.length; i += 1) {
    const name = pool[i];
    if (!usedFlowers.has(name) && !terminals.has(name)) {
      return name;
    }
  }
  const fallback = generateFlowerId();
  return fallback;
}

function ensureChildName(childId, parentId) {
  if (childNames.has(childId)) return childNames.get(childId);
  const name = allocateFlowerForMaster(parentId);
  childNames.set(childId, name);
  usedFlowers.add(name);
  return name;
}

function getChildName(childId) {
  return childNames.get(childId) || childId;
}

function escapeHtml(str) {
  if (!str) return '';
  return str
    .replace(/&/g, '&amp;')
    .replace(/</g, '&lt;')
    .replace(/>/g, '&gt;')
    .replace(/"/g, '&quot;')
    .replace(/'/g, '&#39;');
}

function formatTimeAgo(ts) {
  if (!ts) return '';
  const diffSec = Math.max(1, Math.floor((Date.now() - ts) / 1000));
  if (diffSec < 60) return `${diffSec}s ago`;
  const diffMin = Math.floor(diffSec / 60);
  if (diffMin < 60) return `${diffMin}m ago`;
  const diffHr = Math.floor(diffMin / 60);
  if (diffHr < 24) return `${diffHr}h ago`;
  const diffDay = Math.floor(diffHr / 24);
  return `${diffDay}d ago`;
}

function centerOnElement(el) {
  if (!el || !canvas) return;
  updateCanvasCenterCache();
  const bounds = elementBounds(el);
  const target = fitViewportToBounds(bounds);
  animateViewport(target);
}

function centerOnEntry(entry) {
  if (!entry) return;
  if (!entry.parentId && entry.shell) {
    centerOnElement(entry.shell);
    return;
  }
  const slot = childToSlot.get(entry.id);
  if (slot && slot.element) {
    centerOnElement(slot.element);
  } else if (entry.shell) {
    centerOnElement(entry.shell);
  }
}

function centerOnOldestMaster() {
  const masters = masterEntries()
    .map(([id, entry]) => ({ id, entry }))
    .filter((m) => m.entry && m.entry.shell);
  if (masters.length === 0) return;
  masters.sort((a, b) => {
    const aTs = a.entry.createdAt || 0;
    const bTs = b.entry.createdAt || 0;
    if (aTs === bTs) return 0;
    return aTs - bTs;
  });
  const target = masters[0];
  setActiveTerminal(target.id);
}

function centerOnEntryAfterSidebar(entry, sidebarWasCollapsed) {
  if (!entry) return;
  if (!sidebarWasCollapsed) {
    centerOnEntry(entry);
    return;
  }
  requestAnimationFrame(() => {
    requestAnimationFrame(() => centerOnEntry(entry));
  });
}

function updateSlotTask(slot, prompt) {
  if (!slot) return;
  const text = prompt ? shortenTask(prompt) : 'Task pending';
  if (slot.taskTooltipBody) {
    slot.taskTooltipBody.textContent = text;
  }
  const hasPrompt = Boolean(prompt);
  if (slot.taskButton) {
    slot.taskButton.classList.toggle('hidden', !hasPrompt);
    slot.taskButton.textContent = hasPrompt ? 'See task' : 'No task';
  }
  if (!hasPrompt && slot.taskTooltip) {
    slot.taskTooltip.classList.add('hidden');
  }
}

function formatPathDisplay(p) {
  if (!p) return 'Not set';
  const home = process.env.HOME || '';
  if (home && p.startsWith(home)) {
    return `~${p.slice(home.length) || '/'}`;
  }
  return p;
}

function getWorkingDirectory() {
  if (selectedCwd && selectedCwd.trim()) return selectedCwd;
  return process.cwd ? process.cwd() : '';
}

function folderNameFromPath(p) {
  if (!p) return '';
  const normalized = p.replace(/[\\/]+$/, '') || p;
  const base = path.basename(normalized);
  if (base) return base;
  if (normalized === path.sep) return path.sep;
  return normalized;
}

function setWorkingDirectory(dir) {
  const normalized = (dir || '').trim();
  if (!normalized) return;
  selectedCwd = normalized;
  if (folderPathEl) {
    folderPathEl.textContent = formatPathDisplay(normalized);
    folderPathEl.title = normalized;
  }
  refreshPromptHelper();
}

async function chooseWorkingDirectory() {
  try {
    const result = await ipcRenderer.invoke('select-directory');
    if (!result || result.canceled || !result.path) return;
    setWorkingDirectory(result.path);
  } catch (err) {
    console.error('Failed to choose directory', err);
  }
}

function ensureHostPlaceholder(entry) {
  if (!entry || !entry.hostEl) return null;
  if (!entry.placeholderEl) {
    const el = document.createElement('div');
    el.className = 'terminal-placeholder';
    el.textContent = entry.placeholderText || 'Expand the panel to view this terminal';
    entry.placeholderEl = el;
  } else if (entry.placeholderText) {
    entry.placeholderEl.textContent = entry.placeholderText;
  }
  return entry.placeholderEl;
}

function showHostPlaceholder(entry) {
  const placeholder = ensureHostPlaceholder(entry);
  if (!placeholder || !entry.hostEl) return;
  if (!placeholder.parentElement) {
    entry.hostEl.appendChild(placeholder);
  }
  placeholder.style.display = 'flex';
}

function parkTerminal(entry) {
  if (!entry || !terminalDepot || !entry.terminalDiv) return;
  terminalDepot.appendChild(entry.terminalDiv);
  showHostPlaceholder(entry);
}

function isTerminalInHost(entry) {
  return Boolean(entry && entry.hostEl && entry.hostEl.contains(entry.terminalDiv));
}

function fitAndRefresh(entry) {
  if (!entry || !entry.fitAddon || !entry.term) return;
  const renderService = entry.term._core?.renderService || entry.term._core?._renderService;
  const cssDims = renderService?.dimensions?.css;
  if (cssDims && (!cssDims.cell?.width || !cssDims.cell?.height)) {
    // When terminals are opened inside the hidden depot, xterm may never measure characters.
    renderService._refreshCharSize?.();
  }
  entry.fitAddon.fit();
  entry.term.refresh(0, entry.term.buffer.active.length - 1);
}

function refreshMasterEmptyState() {
  if (!masterEmptyEl) return;
  const hasMasters = masterEntries().length > 0;
  masterEmptyEl.style.display = hasMasters ? 'none' : 'flex';
}

function refreshChildSectionTitle() {
  if (!childSectionTitle) return;
  const count = childPanels.size;
  childSectionTitle.textContent = `${count} Sub Agents`;
}

function refreshChildEmptyState() {
  if (!childEmptyEl) return;
  const hasChildren = childPanels.size > 0;
  childEmptyEl.style.display = hasChildren ? 'none' : 'flex';
  refreshChildSectionTitle();
}

function ensurePlaceholder() {
  if (document.getElementById('placeholder-shell')) return;
  if (!canvasBody) return;
  const shell = document.createElement('div');
  shell.id = 'placeholder-shell';
  shell.className = 'placeholder-shell';
  const label = document.createElement('div');
  label.className = 'placeholder-label';
  label.textContent = '+ Agent Manager';
  shell.appendChild(label);
  const center = canvasCenterWorld();
  const size = 240;
  shell.style.left = `${center.x - size / 2}px`;
  shell.style.top = `${center.y - size / 2}px`;
  shell.addEventListener('click', () => {
    createMaster();
    animatePlaceholderToCorner(shell);
  });
  canvasBody.appendChild(shell);
}

function animatePlaceholderToCorner(shell) {
  if (!shell || !canvasBody) return;
  const rect = canvas.getBoundingClientRect();
  const size = shell.offsetWidth;
  const targetSize = size * 0.5;
  const radiusVisible = size * 0.25; // after scale(0.5)
  const centerOffsetX = targetSize; // half previous horizontal offset
  const centerOffsetY = targetSize * (4 / 3); // move higher: twice the prior offset
  const sidebarWidth = document.getElementById('control-panel')?.offsetWidth || 0;
  shell.classList.add('shrinking');
  shell.style.left = `${sidebarWidth + centerOffsetX - radiusVisible}px`;
  shell.style.top = `${rect.height - centerOffsetY - radiusVisible}px`;
  shell.style.transform = 'scale(0.5)';
  shell.style.opacity = '1';
  shell.classList.add('shrunk');
  const label = shell.querySelector('.placeholder-label');
  if (label) label.textContent = '+';
}

function refreshPlaceholderAnchor() {
  const placeholder = document.getElementById('placeholder-shell');
  if (placeholder && placeholder.classList.contains('shrunk')) {
    animatePlaceholderToCorner(placeholder);
  }
}

function resetPlaceholderToCenter(shell) {
  if (!shell || !canvasBody) return;
  const rect = canvas.getBoundingClientRect();
  const size = shell.offsetWidth;
  const centerX = rect.width / 2 - size / 2;
  const centerY = rect.height / 2 - size / 2;
  shell.classList.add('shrinking');
  shell.style.left = `${centerX}px`;
  shell.style.top = `${centerY}px`;
  shell.style.transform = 'scale(1)';
  shell.style.opacity = '1';
  shell.classList.remove('shrunk');
  const label = shell.querySelector('.placeholder-label');
  if (label) label.textContent = '+ Agent Manager';
}

function updateSubListTitle(masterId) {
  const panel = masterPanels.get(masterId);
  if (!panel || !panel.subListTitle) return;
  const count = Array.from(childPanels.values()).filter((c) => c.masterId === masterId).length;
  panel.subListTitle.textContent = count === 1 ? '1 Sub Agent' : `${count} Sub Agents`;
}

function removeChildPanel(childId) {
  const panel = childPanels.get(childId);
  if (!panel) return;
  if (panel.card && panel.card.parentNode) {
    panel.card.parentNode.removeChild(panel.card);
  }
  if (activeChildPanelId === childId) {
    activeChildPanelId = null;
  }
  childPanels.delete(childId);
  if (panel.masterId) {
    updateSubListTitle(panel.masterId);
  }
  refreshChildEmptyState();
}

function updatePanelCaret(panelObj) {
  if (!panelObj || !panelObj.caret) return;
  const expanded = !panelObj.panel.classList.contains('collapsed');
  panelObj.caret.classList.toggle('rotated', expanded);
}

function ensureMasterPanel(masterId) {
  if (masterPanels.has(masterId)) return masterPanels.get(masterId);
  if (!masterListEl) return null;

  const panel = document.createElement('div');
  panel.className = 'master-panel collapsed';
  panel.dataset.masterId = masterId;

  const header = document.createElement('div');
  header.className = 'master-panel-header';

  const toggleBtn = document.createElement('button');
  toggleBtn.className = 'master-panel-toggle';
  toggleBtn.type = 'button';

  const titles = document.createElement('div');
  titles.className = 'master-panel-titles';

  const titleEl = document.createElement('div');
  titleEl.className = 'master-panel-title';
  titleEl.textContent = 'Untitled manager';

  const subtitleEl = document.createElement('div');
  subtitleEl.className = 'master-panel-subtitle';
  subtitleEl.textContent = masterId;

  const summaryEl = document.createElement('div');
  summaryEl.className = 'master-panel-summary master-panel-summary-empty';
  summaryEl.textContent = 'No final summary yet';

  titles.appendChild(titleEl);
  titles.appendChild(subtitleEl);
  titles.appendChild(summaryEl);

  const caret = document.createElement('div');
  caret.className = 'panel-caret';

  toggleBtn.appendChild(titles);
  toggleBtn.appendChild(caret);

  const deleteBtn = document.createElement('button');
  deleteBtn.className = 'master-panel-delete';
  deleteBtn.type = 'button';
  deleteBtn.setAttribute('aria-label', `Delete ${masterId}`);
  deleteBtn.textContent = '🗑';

  header.appendChild(toggleBtn);
  header.appendChild(deleteBtn);

  const body = document.createElement('div');
  body.className = 'master-panel-body';
  const host = document.createElement('div');
  host.className = 'master-panel-terminal-host';
  body.appendChild(host);

  const subListWrapper = document.createElement('div');
  subListWrapper.className = 'sub-panel-list';
  const subListTitle = document.createElement('div');
  subListTitle.className = 'sub-panel-list-title';
  subListTitle.textContent = 'Sub Agents';
  const subList = document.createElement('div');
  subList.className = 'sub-panel-list-body';
  subListWrapper.appendChild(subListTitle);
  subListWrapper.appendChild(subList);
  body.appendChild(subListWrapper);

  const cwdEl = document.createElement('div');
  cwdEl.className = 'master-panel-cwd';
  cwdEl.textContent = 'Working dir: —';

  panel.appendChild(cwdEl);
  panel.appendChild(header);
  panel.appendChild(body);
  masterListEl.appendChild(panel);

  const panelObj = {
    panel,
    header,
    cwdEl,
    toggleBtn,
    deleteBtn,
    body,
    host,
    titleEl,
    subtitleEl,
    summaryEl,
    caret,
    subList,
    subListTitle,
  };
  masterPanels.set(masterId, panelObj);
  refreshMasterEmptyState();

  toggleBtn.addEventListener('click', () => {
    if (panel.classList.contains('collapsed')) {
      setActiveTerminal(masterId);
    } else {
      collapseMasterPanel(masterId);
    }
  });

  deleteBtn.addEventListener('click', (e) => {
    e.stopPropagation();
    deleteMaster(masterId);
  });

  const existing = terminals.get(masterId);
  if (existing && !existing.parentId) {
    updateMasterPanelCwd(masterId, existing.cwd);
  }

  return panelObj;
}

function renderTerminalInHost(entry, hostEl, opts = {}) {
  if (!entry || !hostEl) return;
  if (entry.placeholderEl && entry.placeholderEl.parentElement === hostEl) {
    entry.placeholderEl.remove();
  }
  const alreadyMounted = hostEl.contains(entry.terminalDiv);
  if (!alreadyMounted) {
    hostEl.innerHTML = '';
    const frame = document.createElement('div');
    frame.className = 'viewer-terminal-frame';
    frame.appendChild(entry.terminalDiv);
    hostEl.appendChild(frame);
  }
  requestAnimationFrame(() => {
    fitAndRefresh(entry);
    if (opts.focus) {
      focusTerm(entry.term, entry.terminalDiv);
    }
  });
}

function expandMasterPanel(masterId, opts = {}) {
  const panel = ensureMasterPanel(masterId);
  const entry = terminals.get(masterId);
  if (!panel || !entry) return;
  const wasExpanded = panel.panel.classList.contains('expanded');
  panel.panel.classList.remove('collapsed');
  panel.panel.classList.add('expanded');
  updatePanelCaret(panel);
  if (!entry.hostEl || entry.hostEl !== panel.host) {
    entry.hostEl = panel.host;
  }
  entry.placeholderText = 'Expand the panel to view this manager terminal';
  const alreadyMounted = wasExpanded && isTerminalInHost(entry);
  if (alreadyMounted) {
    fitAndRefresh(entry);
    if (opts.focus) {
      focusTerm(entry.term, entry.terminalDiv);
    }
  } else {
    renderTerminalInHost(entry, entry.hostEl, { focus: opts.focus });
  }
  if (opts.scroll !== false) {
    panel.panel.scrollIntoView({ behavior: 'smooth', block: 'nearest' });
  }
}

function collapseMasterPanel(masterId) {
  const panel = masterPanels.get(masterId);
  const entry = terminals.get(masterId);
  if (!panel) return;
  panel.panel.classList.add('collapsed');
  panel.panel.classList.remove('expanded');
  updatePanelCaret(panel);
  if (entry) {
    parkTerminal(entry);
  }
  childPanels.forEach((childPanel, childId) => {
    if (childPanel.masterId === masterId) {
      collapseChildPanel(childId);
    }
  });
}

function ensureChildPanel(masterId, childId, label, prompt) {
  ensureMasterPanel(masterId);
  const parentPanel = masterPanels.get(masterId);
  if (!parentPanel || !parentPanel.subList) return null;
  if (childPanels.has(childId)) return childPanels.get(childId);

  const card = document.createElement('div');
  card.className = 'sub-panel collapsed';
  card.dataset.childId = childId;

  const header = document.createElement('div');
  header.className = 'sub-panel-header';

  const titles = document.createElement('div');
  titles.className = 'sub-panel-titles';

  const titleEl = document.createElement('div');
  titleEl.className = 'sub-panel-title';
  titleEl.textContent = getChildName(childId);

  const subtitleEl = document.createElement('div');
  subtitleEl.className = 'sub-panel-subtitle';
  subtitleEl.textContent = shortenTask(prompt) || childId;

  titles.appendChild(titleEl);
  titles.appendChild(subtitleEl);

  const openBtn = document.createElement('button');
  openBtn.type = 'button';
  openBtn.className = 'sub-panel-open';
  openBtn.textContent = 'Open file';

  const body = document.createElement('div');
  body.className = 'sub-panel-body';

  const terminalHost = document.createElement('div');
  terminalHost.className = 'sub-panel-terminal-host';

  body.appendChild(terminalHost);
  header.appendChild(titles);
  header.appendChild(openBtn);
  card.appendChild(header);
  card.appendChild(body);
  parentPanel.subList.appendChild(card);

  const childPanel = {
    card,
    header,
    openBtn,
    titleEl,
    subtitleEl,
    masterId,
    childId,
    body,
    host: terminalHost,
  };
  childPanels.set(childId, childPanel);
  updateSubListTitle(masterId);
  refreshChildEmptyState();

  header.addEventListener('click', () => {
    setActiveTerminal(childId);
  });
  openBtn.addEventListener('click', (e) => {
    e.stopPropagation();
    openFeedbackFile(masterId, childId);
  });

  return childPanel;
}

function collapseChildPanel(childId) {
  const panel = childPanels.get(childId);
  const entry = terminals.get(childId);
  if (!panel) return;
  if (entry) {
    deselectEntry(entry);
  } else if (panel.card) {
    panel.card.classList.remove('selected');
  }
  panel.card.classList.add('collapsed');
  panel.card.classList.remove('expanded');
  if (entry) {
    parkTerminal(entry);
  }
  if (activeChildPanelId === childId) {
    activeChildPanelId = null;
  }
}

function collapseAllChildPanels(exceptId = null) {
  childPanels.forEach((_panel, childId) => {
    if (exceptId && childId === exceptId) return;
    collapseChildPanel(childId);
  });
}

function collapseAllMasterPanels(exceptId = null) {
  masterPanels.forEach((_panel, masterId) => {
    if (exceptId && masterId === exceptId) return;
    collapseMasterPanel(masterId);
  });
}

function expandChildPanel(childId, opts = {}) {
  const panel = childPanels.get(childId);
  const entry = terminals.get(childId);
  if (!panel || !entry) return;
  if (activeChildPanelId && activeChildPanelId !== childId) {
    collapseChildPanel(activeChildPanelId);
  }
  const wasExpanded = panel.card.classList.contains('expanded');
  panel.card.classList.remove('collapsed');
  panel.card.classList.add('expanded');
  entry.hostEl = panel.host || entry.hostEl;
  entry.placeholderText = 'Expand this sub-agent panel to view the terminal';
  const alreadyMounted = wasExpanded && isTerminalInHost(entry);
  if (alreadyMounted) {
    fitAndRefresh(entry);
    if (opts.focus) {
      focusTerm(entry.term, entry.terminalDiv);
    }
  } else {
    renderTerminalInHost(entry, entry.hostEl, { focus: opts.focus });
  }
  activeChildPanelId = childId;
  if (opts.scroll !== false) {
    panel.card.scrollIntoView({ behavior: 'smooth', block: 'nearest' });
  }
}

async function openFeedbackFile(masterId, childId) {
  try {
    const res = await ipcRenderer.invoke('open-feedback-file', { masterId, childId });
    if (!res || res.success !== true) {
      alert(res?.error || 'Could not open feedback file');
    }
  } catch (err) {
    console.error('Failed to open feedback file', err);
    alert('Could not open feedback file');
  }
}

function forceRemoveSlot(slot) {
  if (!slot) return;
  if (slot.statusEl && slot.statusEl.parentNode) slot.statusEl.parentNode.removeChild(slot.statusEl);
  if (slot.element && slot.element.parentNode) slot.element.parentNode.removeChild(slot.element);
  const slots = masterSlots.get(slot.masterId) || [];
  const idx = slots.indexOf(slot);
  if (idx >= 0) slots.splice(idx, 1);
  childToSlot.forEach((v, k) => {
    if (v === slot) childToSlot.delete(k);
  });
}

function deleteMaster(masterId) {
  const masterEntry = terminals.get(masterId);
  const children = Array.from(terminals.values())
    .filter((entry) => entry.parentId === masterId)
    .map((entry) => entry.id);

  // Remove children first
  children.forEach((childId) => {
    const entry = terminals.get(childId);
    const slot = childToSlot.get(childId);
    terminalAPI.close({ id: childId });
    if (entry && entry.term) entry.term.dispose();
    if (entry && entry.terminalDiv && entry.terminalDiv.parentNode) {
      entry.terminalDiv.parentNode.removeChild(entry.terminalDiv);
    }
    if (activeTerminalId === childId) {
      clearActiveTerminalIf(childId);
    }
    if (slot) {
      setSlotStatus(slot, null);
      collapseSlot(slot);
      forceRemoveSlot(slot);
    }
    terminals.delete(childId);
    childToSlot.delete(childId);
    removeChildPanel(childId);
  });

  // Remove master slots
  const slots = masterSlots.get(masterId) || [];
  slots.forEach((slot) => forceRemoveSlot(slot));
  masterSlots.delete(masterId);

  // Remove master terminal
  if (masterEntry) {
    terminalAPI.close({ id: masterId });
    if (masterEntry.term) masterEntry.term.dispose();
    if (masterEntry.terminalDiv && masterEntry.terminalDiv.parentNode) {
      masterEntry.terminalDiv.parentNode.removeChild(masterEntry.terminalDiv);
    }
    if (masterEntry.shell && masterEntry.shell.parentNode) {
      masterEntry.shell.parentNode.removeChild(masterEntry.shell);
    }
    if (activeTerminalId === masterId) {
      clearActiveTerminalIf(masterId);
    }
    terminals.delete(masterId);
  }

  // Remove master title widget
  const titleWidget = masterTitles.get(masterId);
  if (titleWidget && titleWidget.wrapper && titleWidget.wrapper.parentNode) {
    titleWidget.wrapper.parentNode.removeChild(titleWidget.wrapper);
  }
  masterTitles.delete(masterId);
  pendingTitles.delete(masterId);
  const summaryWidget = masterSummaries.get(masterId);
  if (summaryWidget && summaryWidget.box && summaryWidget.box.parentNode) {
    summaryWidget.box.parentNode.removeChild(summaryWidget.box);
  }
  masterSummaries.delete(masterId);
  pendingSummaries.delete(masterId);
  pendingBodies.delete(masterId);

  // Remove panel
  const panel = masterPanels.get(masterId);
  if (panel && panel.panel && panel.panel.parentNode) {
    panel.panel.parentNode.removeChild(panel.panel);
  }
  masterPanels.delete(masterId);

  refreshMasterEmptyState();
  drawEdges();
  releaseFamilyForMaster(masterId);
  updateSubListTitle(masterId);
  const placeholder = document.getElementById('placeholder-shell');
  const remainingMasters = masterEntries();
  if (remainingMasters.length === 0 && placeholder) {
    resetPlaceholderToCenter(placeholder);
  }
  ipcRenderer.invoke('delete-master-temp', { masterId }).catch((err) => {
    console.error('Failed to delete master temp files', err);
  });
}

function selectEntry(entry) {
  if (!entry) return;
  if (entry.shell) entry.shell.classList.add('selected');
  const slot = childToSlot.get(entry.id);
  if (slot && slot.element) {
    slot.element.classList.add('selected');
  }
  if (!entry.parentId) {
    const panel = masterPanels.get(entry.id);
    if (panel && panel.panel) {
      panel.panel.classList.add('selected');
    }
  }
  if (entry.parentId) {
    const childPanel = childPanels.get(entry.id);
    if (childPanel && childPanel.card) {
      childPanel.card.classList.add('selected');
    }
  }
}

function deselectEntry(entry) {
  if (!entry) return;
  if (entry.shell) entry.shell.classList.remove('selected');
  const slot = childToSlot.get(entry.id);
  if (slot && slot.element) {
    slot.element.classList.remove('selected');
  }
  if (!entry.parentId) {
    const panel = masterPanels.get(entry.id);
    if (panel && panel.panel) {
      panel.panel.classList.remove('selected');
    }
  }
  if (entry.parentId) {
    const childPanel = childPanels.get(entry.id);
    if (childPanel && childPanel.card) {
      childPanel.card.classList.remove('selected');
    }
  }
}

function setActiveTerminal(id) {
  const entry = terminals.get(id);
  if (!entry || !entry.terminalDiv) return;
  const isMaster = !entry.parentId;
  const wasCollapsed = sidebarCollapsed;

  expandSidebar();

  if (isMaster) {
    collapseAllChildPanels();
    collapseAllMasterPanels(id);
  } else {
    collapseAllChildPanels(id);
    collapseAllMasterPanels();
  }
  closeAllTaskTooltips();

  if (activeTerminalId === id) {
    if (isMaster) {
      const panel = masterPanels.get(id);
      const isCollapsed = panel ? panel.panel.classList.contains('collapsed') : false;
      if (isCollapsed) {
        expandMasterPanel(id, { focus: true, scroll: true });
      } else if (panel?.host && isTerminalInHost(entry)) {
        fitAndRefresh(entry);
        focusTerm(entry.term, entry.terminalDiv);
        panel.panel.scrollIntoView({ behavior: 'smooth', block: 'nearest' });
      } else {
        expandMasterPanel(id, { focus: true, scroll: true });
      }
    } else {
      const childPanel = childPanels.get(id);
      const isCollapsed = childPanel ? childPanel.card.classList.contains('collapsed') : true;
      if (isCollapsed) {
        expandChildPanel(id, { focus: true });
      } else if (childPanel?.host && isTerminalInHost(entry)) {
        fitAndRefresh(entry);
        focusTerm(entry.term, entry.terminalDiv);
        childPanel.card.scrollIntoView({ behavior: 'smooth', block: 'nearest' });
      } else {
        expandChildPanel(id, { focus: true });
      }
    }
    centerOnEntryAfterSidebar(entry, wasCollapsed);
    return;
  }

  if (activeTerminalId) {
    const prev = terminals.get(activeTerminalId);
    if (prev) {
      deselectEntry(prev);
      parkTerminal(prev);
    }
  }

  activeTerminalId = id;
  selectEntry(entry);
  if (isMaster) {
    expandMasterPanel(id, { focus: true, scroll: true });
  } else {
    expandChildPanel(id, { focus: true });
  }
  centerOnEntryAfterSidebar(entry, wasCollapsed);
}

function clearActiveTerminalIf(id) {
  if (activeTerminalId !== id) return;
  activeTerminalId = null;
  if (activeChildPanelId === id) {
    activeChildPanelId = null;
  }
}

function stopThinking(entry) {
  if (!entry || entry.parentId) return;
  entry.shell.classList.remove('thinking');
  const panel = masterPanels.get(entry.id);
  if (panel?.panel) panel.panel.classList.remove('thinking');
  if (entry.thinkingTimer) {
    clearTimeout(entry.thinkingTimer);
    entry.thinkingTimer = null;
  }
}

function bumpThinking(entry) {
  if (!entry || entry.parentId) return;
  entry.shell.classList.add('thinking');
  const panel = masterPanels.get(entry.id);
  if (panel?.panel) panel.panel.classList.add('thinking');
  if (entry.thinkingTimer) clearTimeout(entry.thinkingTimer);
  entry.thinkingTimer = setTimeout(() => stopThinking(entry), THINKING_IDLE_MS);
}

function sanitizeTitle(text) {
  return (text || '').replace(/\s+/g, ' ').trim();
}

function sanitizeSummary(text) {
  if (typeof text !== 'string') return '';
  const cleaned = text.replace(/\s+/g, ' ').trim();
  if (!cleaned) return '';
  const words = cleaned.split(' ').filter(Boolean);
  if (words.length > 100) {
    return words.slice(0, 100).join(' ');
  }
  return cleaned;
}

function sanitizeBody(text) {
  if (typeof text !== 'string') return '';
  const cleaned = text.replace(/\s+/g, ' ').trim();
  if (!cleaned) return '';
  const words = cleaned.split(' ').filter(Boolean);
  if (words.length > 200) {
    return words.slice(0, 200).join(' ');
  }
  return cleaned;
}

function updateTitlePosition(masterId) {
  const widget = masterTitles.get(masterId);
  const masterEntry = terminals.get(masterId);
  if (!widget || !masterEntry || !masterEntry.shell) return;
  const cx = masterEntry.shell.offsetLeft + masterEntry.shell.offsetWidth / 2;
  const cy = masterEntry.shell.offsetTop + masterEntry.shell.offsetHeight / 2;
  widget.wrapper.style.left = `${cx}px`;
  widget.wrapper.style.top = `${cy}px`;
}

function positionAllTitles() {
  masterTitles.forEach((_, masterId) => updateTitlePosition(masterId));
}

function ensureMasterTitle(masterId) {
  if (masterTitles.has(masterId)) return masterTitles.get(masterId);
  const wrapper = document.createElement('div');
  wrapper.className = 'master-title';
  wrapper.dataset.masterId = masterId;

  const label = document.createElement('div');
  label.className = 'master-title-label';
  label.textContent = masterId;

  const typeRow = document.createElement('div');
  typeRow.className = 'master-title-type-row';
  const typeEl = document.createElement('div');
  typeEl.className = 'master-title-type';
  typeEl.textContent = 'Manager';
  const imageBtn = document.createElement('button');
  imageBtn.type = 'button';
  imageBtn.className = 'name-image-btn';
  imageBtn.innerHTML = '<svg viewBox="0 0 24 24" aria-hidden="true" focusable="false"><path fill="currentColor" d="M5 5h14a2 2 0 0 1 2 2v10a2 2 0 0 1-2 2H5a2 2 0 0 1-2-2V7a2 2 0 0 1 2-2Zm0 2v10h14V7H5Zm2 8 3.5-4.5 2.5 3.01L15 13l2 2.5V15H7Z"/></svg>';
  imageBtn.addEventListener('click', (e) => {
    e.stopPropagation();
    openFlowerSearch(masterId);
  });
  typeRow.appendChild(typeEl);
  typeRow.appendChild(imageBtn);

  const textEl = document.createElement('div');
  textEl.className = 'master-title-text';
  textEl.contentEditable = 'true';
  textEl.spellcheck = false;
  textEl.textContent = 'Untitled manager';
  textEl.setAttribute('aria-label', `Title for ${masterId}`);

  const bodyEl = document.createElement('div');
  bodyEl.className = 'master-title-body master-title-body-empty';
  bodyEl.textContent = 'No body yet';
  bodyEl.setAttribute('aria-label', `Body for ${masterId}`);

  wrapper.appendChild(label);
  wrapper.appendChild(typeRow);
  wrapper.appendChild(textEl);
  wrapper.appendChild(bodyEl);
  world.appendChild(wrapper);

  const handleApply = () => applyMasterTitle(masterId, textEl.textContent || '', 'manual');
  textEl.addEventListener('keydown', (e) => {
    if (e.key === 'Enter') {
      e.preventDefault();
      textEl.blur();
    }
  });
  textEl.addEventListener('input', handleApply);
  textEl.addEventListener('blur', handleApply);

  const widget = { wrapper, textEl, label, bodyEl, bodyLabel: null };
  masterTitles.set(masterId, widget);
  updateTitlePosition(masterId);
  return widget;
}

function applyMasterTitle(masterId, title, source = 'manual') {
  const cleaned = sanitizeTitle(title) || 'Untitled manager';
  const masterExists = terminals.has(masterId) && !terminals.get(masterId).parentId;
  if (!masterExists) {
    pendingTitles.set(masterId, cleaned);
    return;
  }
  const widget = ensureMasterTitle(masterId);
  if (widget.textEl.textContent !== cleaned) {
    widget.textEl.textContent = cleaned;
  }
  const entry = terminals.get(masterId);
  if (entry && !entry.parentId) {
    entry.label = `${cleaned} · ${masterId}`;
    if (entry.shell) entry.shell.title = entry.label;
  }
  const panel = ensureMasterPanel(masterId);
  if (panel) {
    panel.titleEl.textContent = cleaned;
    panel.subtitleEl.textContent = masterId;
    updatePanelCaret(panel);
  }
  document.title = `${cleaned} · AgentGarden`;
  pendingTitles.delete(masterId);
  if (source === 'external') {
    widget.textEl.classList.add('title-ping');
    setTimeout(() => widget.textEl.classList.remove('title-ping'), 400);
  }
}

function applyMasterBody(masterId, body, source = 'manual') {
  const cleaned = sanitizeBody(body);
  const masterExists = terminals.has(masterId) && !terminals.get(masterId).parentId;
  if (!masterExists) {
    if (cleaned) {
      pendingBodies.set(masterId, cleaned);
    } else {
      pendingBodies.delete(masterId);
    }
    return;
  }
  const widget = ensureMasterTitle(masterId);
  if (!widget || !widget.bodyEl) return;
  if (cleaned) {
    widget.bodyEl.textContent = cleaned;
    widget.bodyEl.classList.remove('master-title-body-empty');
  } else {
    widget.bodyEl.textContent = 'No body yet';
    widget.bodyEl.classList.add('master-title-body-empty');
  }
  const entry = terminals.get(masterId);
  if (entry) {
    entry.currentBody = cleaned;
  }
  pendingBodies.delete(masterId);
  if (cleaned && source === 'external') {
    widget.bodyEl.classList.add('body-ping');
    setTimeout(() => widget.bodyEl.classList.remove('body-ping'), 420);
  }
}

function ensureMasterSummary(masterId) {
  if (masterSummaries.has(masterId)) return masterSummaries.get(masterId);
  const entry = terminals.get(masterId);
  if (!entry || entry.parentId || !entry.shell) return null;
  const box = document.createElement('div');
  box.className = 'master-summary hidden';
  const labelEl = document.createElement('div');
  labelEl.className = 'master-summary-label';
  labelEl.textContent = 'Final summary';
  const textEl = document.createElement('div');
  textEl.className = 'master-summary-text';
  textEl.textContent = '';
  box.appendChild(labelEl);
  box.appendChild(textEl);
  entry.shell.appendChild(box);
  const widget = { box, labelEl, textEl };
  masterSummaries.set(masterId, widget);
  return widget;
}

function updateMasterPanelCwd(masterId, cwd) {
  const panel = masterPanels.get(masterId);
  if (!panel || !panel.cwdEl) return;
  const folder = folderNameFromPath(cwd) || 'Unknown';
  panel.cwdEl.textContent = folder;
  panel.cwdEl.title = cwd || 'Unknown location';
}

function applyMasterSummary(masterId, summary, source = 'manual') {
  const cleaned = sanitizeSummary(summary);
  const masterExists = terminals.has(masterId) && !terminals.get(masterId).parentId;
  if (!masterExists) {
    if (cleaned) {
      pendingSummaries.set(masterId, cleaned);
    }
    return;
  }
  const widget = ensureMasterSummary(masterId);
  if (!widget) return;
  const entry = terminals.get(masterId);
  if (cleaned) {
    widget.textEl.textContent = cleaned;
    widget.box.classList.remove('hidden');
  } else {
    widget.textEl.textContent = '';
    widget.box.classList.add('hidden');
  }
  if (entry) {
    entry.finalSummary = cleaned;
  }
  const panel = masterPanels.get(masterId);
  if (panel && panel.summaryEl) {
    if (cleaned) {
      panel.summaryEl.textContent = cleaned;
      panel.summaryEl.classList.remove('master-panel-summary-empty');
    } else {
      panel.summaryEl.textContent = 'No final summary yet';
      panel.summaryEl.classList.add('master-panel-summary-empty');
    }
  }
  pendingSummaries.delete(masterId);
  if (cleaned && source === 'external' && widget.box) {
    widget.box.classList.add('summary-ping');
    setTimeout(() => widget.box.classList.remove('summary-ping'), 420);
  }
}

function pickOrCreateMaster(requestedId) {
  const masters = masterEntries();
  const existing = requestedId ? masters.find(([id]) => id === requestedId) : null;
  if (existing) {
    return { id: existing[0], created: false, note: '' };
  }
  if (requestedId && masters.length > 0) {
    const fallback = masters[masters.length - 1][0];
    return {
      id: fallback,
      created: false,
      note: ` (requested ${requestedId} not found; using ${fallback})`,
    };
  }
  if (requestedId) {
    const familyKey = ensureFamilyForMaster(requestedId) || pickFamilyKey();
    if (!familyKey) {
    alert('All flower families are currently in use. Close a manager to add a new one.');
      const fallback = masters.length > 0 ? masters[masters.length - 1][0] : null;
      return { id: fallback || requestedId, created: false, note: '' };
    }
    if (!familyAssignments.has(requestedId)) {
      familyAssignments.set(requestedId, familyKey);
      usedFamilies.add(familyKey);
    }
    createTerminal({ id: requestedId, label: `${requestedId} · shell (auto-created)` });
    ensureSlots(requestedId);
    updateTitlePosition(requestedId);
    return { id: requestedId, created: true, note: '' };
  }
  if (masters.length > 0) {
    const fallback = masters[masters.length - 1][0];
    return { id: fallback, created: false, note: '' };
  }
  const familyKey = pickFamilyKey();
  if (!familyKey) {
    alert('All flower families are currently in use. Close a manager to add a new one.');
    return { id: null, created: false, note: '' };
  }
  const newId = terminals.has(familyKey) ? `${familyKey}-${Date.now().toString(36)}` : familyKey;
  familyAssignments.set(newId, familyKey);
  usedFamilies.add(familyKey);
  createTerminal({ id: newId, label: `${newId} · shell (auto-created)` });
  ensureSlots(newId);
  updateTitlePosition(newId);
  return { id: newId, created: true, note: '' };
}

function centerPosition(element) {
  const center = canvasCenterWorld();
  const x = center.x - element.offsetWidth / 2;
  const y = center.y - element.offsetHeight / 2;
  return { x, y };
}

applyTransform();

function getStackPosition(index) {
  const margin = 20;
  return { x: margin + 40 * index, y: margin + 40 * index };
}

function masterEntries() {
  return Array.from(terminals.entries()).filter(([, v]) => !v.parentId);
}

function shortenTask(prompt) {
  if (!prompt) return 'No task provided';
  const firstLine = prompt
    .split('\n')
    .map((line) => line.trim())
    .find((line) => line.length > 0);
  const base = firstLine || prompt;
  const words = base
    .replace(/\s+/g, ' ')
    .trim()
    .split(' ')
    .slice(0, 10);
  return words.join(' ');
}

function createSlot(masterId, index) {
  const slotEl = document.createElement('div');
  slotEl.className = 'sub-slot occupied collapsed';
  slotEl.dataset.masterId = masterId;
  slotEl.dataset.slotIndex = index;

  const thinkingGlow = document.createElement('div');
  thinkingGlow.className = 'sub-thinking-glow';

  const thinkingRing = document.createElement('div');
  thinkingRing.className = 'sub-thinking-ring';
  const thinkingDot = document.createElement('div');
  thinkingDot.className = 'sub-thinking-dot';
  thinkingRing.appendChild(thinkingDot);

  const face = document.createElement('div');
  face.className = 'slot-face';
  face.innerHTML =
    '<div class="slot-name"></div><div class="slot-role">Sub Agent</div><button class="name-image-btn slot-image-btn" type="button" aria-label="View flower images"><svg viewBox="0 0 24 24" aria-hidden="true" focusable="false"><path fill="currentColor" d="M5 5h14a2 2 0 0 1 2 2v10a2 2 0 0 1-2 2H5a2 2 0 0 1-2-2V7a2 2 0 0 1 2-2Zm0 2v10h14V7H5Zm2 8 3.5-4.5 2.5 3.01L15 13l2 2.5V15H7Z"/></svg></button>';

  const task = document.createElement('div');
  task.className = 'slot-task';
  const taskButton = document.createElement('button');
  taskButton.type = 'button';
  taskButton.className = 'slot-task-button';
  taskButton.textContent = 'See task';
  const taskTooltip = document.createElement('div');
  taskTooltip.className = 'feedback-tooltip task-tooltip tooltip-box hidden';
  const taskTooltipBody = document.createElement('div');
  taskTooltipBody.className = 'feedback-body';
  taskTooltipBody.textContent = 'Task pending';
  taskTooltip.appendChild(taskTooltipBody);
  task.appendChild(taskButton);
  task.appendChild(taskTooltip);

  const terminalWrapper = document.createElement('div');
  terminalWrapper.className = 'slot-terminal';

  slotEl.appendChild(thinkingGlow);
  slotEl.appendChild(thinkingRing);
  slotEl.appendChild(face);
  slotEl.appendChild(task);
  slotEl.appendChild(terminalWrapper);

  world.appendChild(slotEl);
  const slot = {
    id: `slot-${masterId}-${index}`,
    element: slotEl,
    face,
    task,
    terminalWrapper,
    occupiedBy: null,
    masterId,
    expanded: false,
    status: null,
    statusEl: null,
    taskButton,
    taskTooltip,
    taskTooltipBody,
  };
  ensureStatusEl(slot);
  setSlotStatus(slot, null);
  updateSlotTask(slot, null);

  const toggleTask = (e) => {
    e.stopPropagation();
    const isOpen = !taskTooltip.classList.contains('hidden');
    closeAllTaskTooltips(slot.id);
    if (!isOpen) {
      taskTooltip.classList.remove('hidden');
      taskButton.classList.add('open');
    }
  };
  taskButton.addEventListener('click', toggleTask);

  slotEl.addEventListener('click', (e) => {
    e.stopPropagation();
    if (!slot.occupiedBy) return;
    setActiveTerminal(slot.occupiedBy);
  });

  return slot;
}

function ensureSlots(masterId) {
  if (!masterSlots.has(masterId)) {
    masterSlots.set(masterId, []);
  }
  return masterSlots.get(masterId);
}

function findAvailableSlot(masterId) {
  const slots = ensureSlots(masterId);
  const masterEntry = terminals.get(masterId);
  if (!masterEntry) return null;
  const slot = createSlot(masterId, slots.length);
  slots.push(slot);
  // start at master center, then animate outward
  const center = circleInfo(masterEntry.shell);
  slot.element.style.left = `${center.x}px`;
  slot.element.style.top = `${center.y}px`;
  slot.element.classList.add('spawning');
  requestAnimationFrame(() => {
    positionSlots(masterId);
    slot.element.classList.remove('spawning');
  });
  return slot;
}

function removeSlot(slot) {
  if (!slot) return;
  const childId = slot.occupiedBy;
  const masterId = slot.masterId;
  const slots = masterSlots.get(masterId) || [];
  const masterEntry = terminals.get(masterId);
  let finished = false;
  const removeNow = () => {
    if (finished) return;
    finished = true;
    slot.occupiedBy = null;
    slot.expanded = false;
    setSlotStatus(slot, null);
    if (slot.statusEl && slot.statusEl.parentNode) slot.statusEl.parentNode.removeChild(slot.statusEl);
    if (slot.taskTooltip && slot.taskTooltip.parentNode) slot.taskTooltip.parentNode.removeChild(slot.taskTooltip);
    if (slot.taskButton && slot.taskButton.parentNode) slot.taskButton.parentNode.removeChild(slot.taskButton);
    if (slot.element && slot.element.parentNode) slot.element.parentNode.removeChild(slot.element);
    const idx = slots.indexOf(slot);
    if (idx >= 0) slots.splice(idx, 1);
    childToSlot.forEach((v, k) => {
      if (v === slot) childToSlot.delete(k);
    });
    if (childId) {
      const friendly = childNames.get(childId);
      if (friendly) usedFlowers.delete(friendly);
      childNames.delete(childId);
    }
    positionSlots(masterId);
    drawEdges();
    if (childId) {
      feedbackMarkers.delete(childId);
      feedbackLogs.delete(childId);
    }
  };

  if (!masterEntry || !masterEntry.shell) {
    removeNow();
    return;
  }
  const center = circleInfo(masterEntry.shell);
  slot.element.classList.add('collapsing');
  slot.element.classList.remove('expanded');
  slot.element.classList.add('collapsed');
  slot.element.style.left = `${center.x}px`;
  slot.element.style.top = `${center.y}px`;
  setSlotStatus(slot, null);
  const onDone = () => {
    slot.element.removeEventListener('transitionend', onDone);
    removeNow();
  };
  slot.element.addEventListener('transitionend', onDone);
  setTimeout(onDone, 260);
}

function positionSlots(masterId) {
  const slots = masterSlots.get(masterId);
  const masterEntry = terminals.get(masterId);
  if (!slots || !masterEntry) return;
  const rect = {
    width: masterEntry.shell.offsetWidth,
    height: masterEntry.shell.offsetHeight,
    left: masterEntry.shell.offsetLeft,
    top: masterEntry.shell.offsetTop,
  };
  const centerX = rect.left + rect.width / 2;
  const centerY = rect.top + rect.height / 2;
  const radius = Math.max(rect.width, rect.height) / 2 + DEFAULT_ORBIT;
  slots.forEach((slot, idx) => {
    if (slot.element.classList.contains('collapsing')) return;
    const angle = (idx / slots.length) * Math.PI * 2 - Math.PI / 2;
    const x = centerX + Math.cos(angle) * radius;
    const y = centerY + Math.sin(angle) * radius;
    slot.element.style.left = `${x}px`;
    slot.element.style.top = `${y}px`;
  });
  updateTitlePosition(masterId);
  drawEdges();
}

function expandSlot(slot) {
  if (!slot || !slot.occupiedBy) return;
  slot.expanded = true;
  slot.element.classList.remove('collapsed');
  slot.element.classList.add('expanded');
  const entry = terminals.get(slot.occupiedBy);
  if (entry) {
    setTimeout(() => {
      fitAndRefresh(entry);
      focusTerm(entry.term, entry.terminalDiv);
    }, 30);
  }
}

function collapseSlot(slot) {
  if (!slot) return;
  slot.expanded = false;
  slot.element.classList.add('collapsed');
  slot.element.classList.remove('expanded');
}

function circleInfo(el, pad = 0) {
  const w = el.offsetWidth;
  const h = el.offsetHeight;
  const centeredByTranslate = el.classList.contains('sub-slot');
  // Sub-slot circles are centered via translate(-50%, -50%), so their CSS left/top already represent the center.
  const x = centeredByTranslate ? el.offsetLeft : el.offsetLeft + w / 2;
  const y = centeredByTranslate ? el.offsetTop : el.offsetTop + h / 2;
  return {
    x,
    y,
    r: Math.max(Math.max(w, h) / 2 - pad, 0),
  };
}

function computeMasterWebRadius(masterId) {
  const entry = terminals.get(masterId);
  if (!entry) return 0;
  const masterCircle = circleInfo(entry.shell, 0);
  const slots = masterSlots.get(masterId) || [];
  let radius = masterCircle.r;
  slots.forEach((slot) => {
    const slotCircle = circleInfo(slot.element, 0);
    const dist = Math.hypot(slotCircle.x - masterCircle.x, slotCircle.y - masterCircle.y);
    radius = Math.max(radius, dist + slotCircle.r);
  });
  return radius;
}

function estimateNewMasterRadius() {
  const existing = masterEntries();
  if (existing.length > 0) {
    return Math.max(...existing.map(([id]) => computeMasterWebRadius(id)));
  }
  const shellRadius = DEFAULT_SHELL_SIZE / 2;
  const slotRadius = DEFAULT_SLOT_SIZE / 2;
  const orbit = shellRadius + DEFAULT_ORBIT;
  return orbit + slotRadius;
}

function isPositionFreeForMaster(pos, newRadius) {
  return masterEntries().every(([id, entry]) => {
    const center = circleInfo(entry.shell, 0);
    const radius = computeMasterWebRadius(id);
    const dist = Math.hypot(pos.x - center.x, pos.y - center.y);
    return dist >= radius + newRadius + MASTER_BUFFER;
  });
}

function findFreeMasterPosition(newRadius) {
  const base = canvasCenterWorld();
  if (isPositionFreeForMaster(base, newRadius)) {
    return base;
  }

  const step = newRadius * 2 + MASTER_BUFFER;
  const maxRings = 18;
  for (let ring = 1; ring <= maxRings; ring += 1) {
    const ringRadius = ring * step;
    const points = Math.max(8, ring * 10);
    for (let i = 0; i < points; i += 1) {
      const angle = (i / points) * Math.PI * 2;
      const candidate = {
        x: base.x + Math.cos(angle) * ringRadius,
        y: base.y + Math.sin(angle) * ringRadius,
      };
      if (isPositionFreeForMaster(candidate, newRadius)) {
        return candidate;
      }
    }
  }
  return base;
}

function ensureStatusEl(slot) {
  if (!slot.statusEl) {
    const el = document.createElement('div');
    el.className = 'status-badge hidden';
    slot.statusEl = el;
    slot.element.appendChild(el);
  }
  return slot.statusEl;
}

function emitPetalBurst(slot) {
  if (!slot || !slot.element) return;
  const burst = document.createElement('div');
  burst.className = 'petal-confetti';
  const count = 12 + Math.floor(Math.random() * 6);
  for (let i = 0; i < count; i += 1) {
    const petal = document.createElement('span');
    petal.className = 'petal';
    const dx = (Math.random() * 160 - 80).toFixed(1);
    const dy = (Math.random() * 120 - 40).toFixed(1);
    const rot = (Math.random() * 160 - 80).toFixed(1);
    const dur = (1.8 + Math.random() * 0.8).toFixed(2);
    const delay = (Math.random() * 0.2).toFixed(2);
    petal.style.setProperty('--dx', `${dx}px`);
    petal.style.setProperty('--dy', `${dy}px`);
    petal.style.setProperty('--rot', `${rot}deg`);
    petal.style.setProperty('--dur', `${dur}s`);
    petal.style.setProperty('--delay', `${delay}s`);
    burst.appendChild(petal);
  }
  slot.element.appendChild(burst);
  setTimeout(() => {
    if (burst && burst.parentNode) {
      burst.parentNode.removeChild(burst);
    }
  }, 2400);
}

function setSlotStatus(slot, status) {
  ensureStatusEl(slot);
  const prevStatus = slot.status;
  slot.status = status || null;
  slot.element.classList.remove('thinking', 'waiting', 'update');
  const childPanel = childPanels.get(slot.occupiedBy);
  if (childPanel?.card) {
    childPanel.card.classList.remove('thinking', 'waiting', 'update');
  }
  const el = slot.statusEl;
  if (!slot.status) {
    el.className = 'status-badge hidden';
    el.innerHTML = '';
    return;
  }
  el.className = 'status-badge';
  if (status === 'thinking') {
    slot.element.classList.add('thinking');
    if (childPanel?.card) childPanel.card.classList.add('thinking');
    el.innerHTML = `
      <div class="status-chip">
        <div class="chip-text">
          <div class="chip-label">Growing idea</div>
          <div class="chip-sub">Sub-agent is thinking</div>
        </div>
        <div class="chip-pips">
          <span></span><span></span><span></span>
        </div>
        <div class="chip-actions"></div>
      </div>
    `;
  } else if (status === 'update') {
    slot.element.classList.add('update');
    if (childPanel?.card) childPanel.card.classList.add('update');
    el.innerHTML = `
      <div class="status-chip status-update">
        <div class="chip-text">
          <div class="chip-label">Progress sent</div>
          <div class="chip-sub">Child shared an update</div>
        </div>
        <div class="chip-pips return">
          <span></span><span></span><span></span>
        </div>
        <div class="chip-actions"></div>
      </div>
    `;
  } else if (status === 'waiting') {
    slot.element.classList.add('waiting');
    if (childPanel?.card) childPanel.card.classList.add('waiting');
    el.innerHTML = `
      <div class="status-chip status-waiting">
        <div class="chip-text">
          <div class="chip-label">Waiting on manager</div>
          <div class="chip-sub">Needs feedback to continue</div>
        </div>
        <div class="chip-pips return">
          <span></span><span></span><span></span>
        </div>
        <div class="chip-actions"></div>
      </div>
    `;
    if (prevStatus !== 'waiting') {
      emitPetalBurst(slot);
    }
  } else {
    el.innerHTML = '';
    el.className = 'status-badge hidden';
  }
  attachFeedbackMarker(slot);
  widenStatusChip(slot);
  placeStatus(slot);
  drawEdges();
}

function placeStatus(slot) {
  if (!slot.status || !slot.statusEl) return;
  const el = slot.statusEl;
  el.style.left = '50%';
  el.style.top = '0';
}

function handleWheelZoom(e) {
  // Let wheel events inside terminals scroll the terminal rather than zoom
  if (e.target.closest('.terminal')) return;
  stopPanAnimation();
  updateCanvasCenterCache();
  // allow canvas zooming when not interacting with terminals
  const direction = e.deltaY < 0 ? 1 : -1;
  const factor = direction > 0 ? 1.08 : 0.92;
  const newScale = clamp(viewState.scale * factor, MIN_SCALE, MAX_SCALE);
  if (newScale === viewState.scale) return;

  const rect = canvas.getBoundingClientRect();
  const focusWorld = screenToWorld(e.clientX, e.clientY);
  viewState.scale = newScale;
  viewState.tx = e.clientX - rect.left - focusWorld.x * newScale;
  viewState.ty = e.clientY - rect.top - focusWorld.y * newScale;
  applyTransform();
  e.preventDefault();
}

function handlePanStart(e) {
  if (
    e.target.closest('.master-shell') ||
    e.target.closest('.sub-slot') ||
    e.target.closest('.master-title') ||
    e.target.closest('.placeholder-shell')
  ) {
    return;
  }
  stopPanAnimation();
  isPanning = true;
  panEngaged = false;
  panStart = { x: e.clientX, y: e.clientY };
  panOrigin = { tx: viewState.tx, ty: viewState.ty };
  updateCanvasCenterCache();
  canvas.style.cursor = 'grabbing';
  e.preventDefault();
}

function handlePanMove(e) {
  if (!isPanning) return;
  const dx = e.clientX - panStart.x;
  const dy = e.clientY - panStart.y;
  // Only engage panning (and collapse sidebar) after a small movement to avoid
  // swallowing simple clicks.
  if (!panEngaged) {
    const dist = Math.hypot(dx, dy);
    if (dist < 3) return;
    panEngaged = true;
    collapseSidebar({ skipOffset: true });
  }
  viewState.tx = panOrigin.tx + dx;
  viewState.ty = panOrigin.ty + dy;
  applyTransform();
  updateCanvasCenterCache();
  e.preventDefault();
}

function handlePanEnd() {
  if (!isPanning) return;
  isPanning = false;
  panEngaged = false;
  canvas.style.cursor = 'grab';
  updateCanvasCenterCache();
  scheduleRecenterOnLayout();
}

function attachFeedbackMarker(slot) {
  if (!slot || !slot.statusEl || !slot.occupiedBy) return;
  const chip = slot.statusEl.querySelector('.status-chip');
  const actions = chip?.querySelector('.chip-actions');
  if (!chip || !actions) return;
  const entry = ensureFeedbackMarker(slot.occupiedBy);
  if (!entry) return;
  if (!actions.contains(entry.marker)) {
    actions.appendChild(entry.marker);
  }
}

function widenStatusChip(slot) {
  if (!slot?.statusEl) return;
  const chip = slot.statusEl.querySelector('.status-chip');
  if (!chip) return;
  chip.style.minWidth = 'auto';
  chip.style.width = 'auto';
  requestAnimationFrame(() => {
    const baseWidth = chip.getBoundingClientRect().width || chip.scrollWidth;
    if (baseWidth) {
      chip.style.minWidth = `${baseWidth * 2}px`;
    }
  });
}

function focusTerm(term, terminalDiv) {
  if (terminalDiv && terminalDiv.isConnected) {
    terminalDiv.focus();
  }
  term.focus();
}

function shellQuotePath(p) {
  if (!p) return '';
  return `'${p.replace(/'/g, "'\\''")}'`;
}

function loadAgentPathOverrides() {
  try {
    const raw = localStorage.getItem(AGENT_PATH_STORAGE_KEY);
    if (!raw) return {};
    const parsed = JSON.parse(raw);
    return parsed && typeof parsed === 'object' ? parsed : {};
  } catch (err) {
    console.warn('Failed to load agent path overrides', err);
    return {};
  }
}

function persistAgentPathOverrides() {
  try {
    localStorage.setItem(AGENT_PATH_STORAGE_KEY, JSON.stringify(agentPathOverrides));
  } catch (err) {
    console.warn('Failed to persist agent path overrides', err);
  }
}

function syncAgentPathInputs() {
  agentPathInputs.forEach((inputs, agentId) => {
    inputs.forEach((input) => {
      if (input) input.value = agentPathOverrides[agentId] || '';
    });
  });
}

function isExecutableFile(p) {
  try {
    const stat = fs.statSync(p);
    if (!stat.isFile()) return false;
    if (process.platform === 'win32') return true;
    return Boolean(stat.mode & 0o111);
  } catch (err) {
    return false;
  }
}

function detectAgentPath(option) {
  const override = agentPathOverrides[option.id];
  if (override && isExecutableFile(override)) {
    return override;
  }
  const whichCmd = process.platform === 'win32' ? 'where' : 'which';
  try {
    const result = spawnSync(whichCmd, [option.binary], { encoding: 'utf8' });
    if (result.status === 0) {
      const candidate = (result.stdout || '')
        .split(/\r?\n/)
        .map((line) => line.trim())
        .find(Boolean);
      if (candidate && isExecutableFile(candidate)) {
        return candidate;
      }
    }
  } catch (err) {
    console.warn(`Failed to detect agent binary "${option.binary}"`, err);
  }
  return null;
}

function getAgentOption(agentId) {
  return AGENT_OPTIONS.find((opt) => opt.id === agentId) || AGENT_OPTIONS[0];
}

function persistSelectedAgent(agentId) {
  if (!window?.localStorage) return;
  try {
    localStorage.setItem(AGENT_STORAGE_KEY, agentId);
  } catch (err) {
    // ignore storage failures
  }
}

function loadSelectedAgent() {
  if (!window?.localStorage) return null;
  try {
    return localStorage.getItem(AGENT_STORAGE_KEY);
  } catch (err) {
    return null;
  }
}

function ensureSelectedAgentValid() {
  const stored = loadSelectedAgent();
  if (stored && AGENT_OPTIONS.some((opt) => opt.id === stored)) {
    selectedAgent = stored;
  }
  if (agentAvailability[selectedAgent]) return;
  const fallback = AGENT_OPTIONS.find((opt) => agentAvailability[opt.id]);
  if (fallback) {
    selectedAgent = fallback.id;
    persistSelectedAgent(selectedAgent);
  }
}

function updateAgentButtons() {
  agentButtons.forEach((btn) => {
    const agentId = btn.dataset.agentOption;
    if (!agentId) return;
    const option = getAgentOption(agentId);
    const available = Boolean(agentAvailability[agentId]);
    const isSelected = agentId === selectedAgent && available;
    btn.classList.toggle('selected', isSelected);
    btn.classList.toggle('unavailable', !available);
    btn.disabled = !available;
    btn.setAttribute('aria-disabled', String(!available));
    btn.setAttribute('aria-selected', String(isSelected));
    btn.title = available
      ? `${option.label} detected`
      : `${option.label} not detected on PATH`;
  });
  agentDefaultInputs.forEach((input) => {
    input.checked = input.value === selectedAgent;
  });
  const detected = AGENT_OPTIONS.filter((opt) => agentAvailability[opt.id]).map((opt) => opt.label);
  const statusText = detected.length ? `Detected: ${detected.join(', ')}` : 'No local agents detected';
  agentStatusEls.forEach((el) => {
    el.textContent = statusText;
    el.classList.toggle('error', detected.length === 0);
  });
}

function refreshAgentDetection() {
  const nextAvailability = {};
  const nextResolved = {};
  AGENT_OPTIONS.forEach((opt) => {
    const resolved = detectAgentPath(opt);
    nextResolved[opt.id] = resolved;
    nextAvailability[opt.id] = Boolean(resolved);
  });
  agentResolvedPaths = nextResolved;
  agentAvailability = nextAvailability;
  ensureSelectedAgentValid();
  updateAgentButtons();
}

function setSelectedAgent(agentId) {
  const option = getAgentOption(agentId);
  if (!option) return;
  selectedAgent = option.id;
  persistSelectedAgent(selectedAgent);
  updateAgentButtons();
}

function getSelectedAgentOption() {
  return getAgentOption(selectedAgent);
}

function getAgentBootstrap() {
  const option = getSelectedAgentOption();
  const available = Boolean(agentAvailability[option.id]);
  const resolvedPath = agentResolvedPaths[option.id];
  let command = null;
  if (available) {
    if (option.id === 'codex') {
      command = resolvedPath ? `${shellQuotePath(resolvedPath)} --yolo` : 'codex --yolo';
    } else if (option.id === 'claude') {
      command = resolvedPath ? shellQuotePath(resolvedPath) : 'claude';
    } else if (option.id === 'gemini') {
      command = resolvedPath ? shellQuotePath(resolvedPath) : 'gemini';
    }
  }
  return { option, command, available };
}

function openSettings() {
  if (!settingsOverlay) return;
  refreshAgentDetection();
  settingsOverlay.classList.remove('hidden');
}

function closeSettings() {
  if (!settingsOverlay) return;
  settingsOverlay.classList.add('hidden');
}

function bootstrapTerminal(id, cwd) {
  const bootstrap = [];
  if (cwd) {
    bootstrap.push(`cd ${shellQuotePath(cwd)}`);
  }
  const agent = getAgentBootstrap();
  bootstrap.push('clear');
  bootstrap.push(`echo '=== ${agent.option.label} ==='`);
  if (agent.available && agent.command) {
    bootstrap.push(agent.command);
  } else {
    bootstrap.push(`echo '${agent.option.label} CLI not detected on PATH'`);
  }
  bootstrap.forEach((line) => terminalAPI.send({ id, data: `${line}\n` }));
}

function createTerminal({ id, label, parentId = null, prompt, slot = null, position = null }) {
  if (terminals.has(id)) {
    // Avoid duplicating the same terminal id; clean up any unused slot passed in.
    if (slot && slot.element && !slot.occupiedBy) {
      removeSlot(slot);
    }
    return id;
  }
  const isChild = Boolean(parentId);
  if (isChild && !slot) {
    console.error('Missing slot for child terminal', id);
    return null;
  }

  let shell;
  let header;
  let masterPanelHost = null;
  let childPanel = null;
  const cwd = getWorkingDirectory();

  if (isChild) {
    ensureChildName(id, parentId);
  } else {
    const familyKey = ensureFamilyForMaster(id);
    if (!familyKey) {
      console.warn('No available family for manager; creation skipped.');
      return null;
    }
  }

  if (isChild && parentId) {
    childPanel = ensureChildPanel(parentId, id, label, prompt);
  }

  if (isChild && slot) {
    shell = slot.element;
    slot.occupiedBy = id;
    slot.element.classList.remove('placeholder');
    slot.element.classList.add('occupied', 'collapsed');
    const displayName = getChildName(id);
    const nameEl = slot.face.querySelector('.slot-name');
    const roleEl = slot.face.querySelector('.slot-role');
    const imgBtn = slot.face.querySelector('.slot-image-btn');
    if (nameEl) nameEl.textContent = displayName;
    if (roleEl) roleEl.textContent = 'Sub Agent';
    if (imgBtn) {
      imgBtn.onclick = (e) => {
        e.stopPropagation();
        openFlowerSearch(displayName);
      };
    }
    updateSlotTask(slot, prompt);
    slot.terminalWrapper.innerHTML = '';
    setSlotStatus(slot, 'thinking');
  } else {
    shell = document.createElement('div');
    shell.className = 'master-shell';
    shell.dataset.terminalId = id;
    shell.title = label;
    const thinkingGlow = document.createElement('div');
    thinkingGlow.className = 'thinking-glow';
    const thinkingRing = document.createElement('div');
    thinkingRing.className = 'thinking-ring';
    const thinkingDot = document.createElement('div');
    thinkingDot.className = 'thinking-dot';
    thinkingRing.appendChild(thinkingDot);
    header = shell; // use the shell itself as drag handle
    shell.appendChild(thinkingGlow);
    shell.appendChild(thinkingRing);
    world.appendChild(shell);
    ensureMasterTitle(id);
    const panel = ensureMasterPanel(id);
    if (panel) {
      masterPanelHost = panel.host;
      panel.subtitleEl.textContent = id;
      updatePanelCaret(panel);
    }
  }

  const hostEl = isChild ? childPanel?.host || slot?.terminalWrapper : masterPanelHost;

  const terminalDiv = document.createElement('div');
  terminalDiv.className = isChild ? 'terminal sub-terminal' : 'terminal';
  terminalDiv.tabIndex = 0;
  if (terminalDepot) {
    terminalDepot.appendChild(terminalDiv);
  }

  if (!isChild) {
    const newRadius = estimateNewMasterRadius();
    const targetCenter = position || findFreeMasterPosition(newRadius);
    const shellWidth = shell.offsetWidth || DEFAULT_SHELL_SIZE;
    const shellHeight = shell.offsetHeight || DEFAULT_SHELL_SIZE;
    const left = (targetCenter ? targetCenter.x : centerPosition(shell).x) - shellWidth / 2;
    const top = (targetCenter ? targetCenter.y : centerPosition(shell).y) - shellHeight / 2;
    shell.style.left = `${left}px`;
    shell.style.top = `${top}px`;
  }

  const term = new Terminal({
    convertEol: true,
    cursorBlink: true,
    fontFamily: 'Menlo, Monaco, "Courier New", monospace',
    fontSize: 12,
    theme: {
      background: '#081020',
      foreground: '#eef6ff',
      cursor: '#ff9ae4',
      cursorAccent: '#050910',
      selection: 'rgba(255, 141, 220, 0.22)',
      black: '#050910',
      green: '#3ee48a',
      magenta: '#ff8ddc',
      cyan: '#5cf4e8',
    },
  });
  const fitAddon = new FitAddon();
  term.loadAddon(fitAddon);
  term.open(terminalDiv);

  // Prevent scroll from bubbling to canvas
  terminalDiv.addEventListener('wheel', (e) => {
    e.stopPropagation();
  });
  ['click', 'mousedown', 'mouseup', 'wheel'].forEach((evt) => {
    terminalDiv.addEventListener(evt, () => focusTerm(term, terminalDiv));
  });

  const helperText = isChild ? null : buildMasterHelperPrompt(id);

  const entry = {
    id,
    label,
    term,
    fitAddon,
    shell,
    terminalDiv,
    hostEl,
    placeholderEl: null,
    parentId,
    slot,
    pendingPrompt: prompt || null,
    promptSent: false,
    readyForPrompt: false,
    fitTimer: null,
    thinkingTimer: null,
    helperText,
    helperSent: isChild ? true : false,
    placeholderText: isChild
      ? 'Expand this sub-agent panel to view the terminal'
      : 'Expand the panel to view this manager terminal',
    finalSummary: '',
    currentBody: '',
    cwd,
    createdAt: Date.now(),
  };

  terminals.set(id, entry);
  refreshMasterEmptyState();

  if (!isChild) {
    ensureMasterSummary(id);
    updateMasterPanelCwd(id, cwd);
    lastMasterId = id;
    refreshPromptHelper();
    const pendingTitle = pendingTitles.get(id);
    if (pendingTitle) {
      applyMasterTitle(id, pendingTitle, 'external');
    } else {
      applyMasterTitle(id, 'Untitled manager', 'manual');
    }
    const pendingSummary = pendingSummaries.get(id);
    if (pendingSummary) {
      applyMasterSummary(id, pendingSummary, 'external');
    }
    const pendingBody = pendingBodies.get(id);
    if (pendingBody) {
      applyMasterBody(id, pendingBody, 'external');
    }
    updateTitlePosition(id);
  }

  if (slot) {
    childToSlot.set(id, slot);
  }

  // Park the terminal off-canvas by default and activate the first created terminal.
  parkTerminal(entry);
  if (!activeTerminalId) {
    setActiveTerminal(id);
  }

  const env = {
    HAM_MASTER_ID: parentId || id,
    HAM_PARENT_ID: parentId || '',
    HAM_CHILD_ID: isChild ? id : '',
    HAM_AGENT_ROLE: isChild ? 'child' : 'master',
    HAM_AGENT_RUNTIME: selectedAgent || '',
    HAM_AGENT_AVAILABLE: agentAvailability[selectedAgent] ? '1' : '0',
  };

  terminalAPI.create({ id, cwd, env }).then((res) => {
    if (!res || res.success !== true) {
      term.write(`\r\n[failed to launch: ${res?.error || 'unknown error'}]\r\n`);
      return;
    }
    bootstrapTerminal(id, cwd);
    // prompt will be sent once initial output arrives
  });

  term.onData((data) => {
    terminalAPI.send({ id, data });
  });

  terminalAPI.onData((_, payload) => {
      if (payload.id === id) {
        term.write(payload.data);
        scrollToBottom();
        const latestEntry = terminals.get(id);
        if (!latestEntry) return;
        bumpThinking(latestEntry);

        // detect readiness via context banner
        if (!latestEntry.readyForPrompt) {
          if (
            /context left/i.test(payload.data) ||
            /To get started/i.test(payload.data) ||
            Boolean(payload.data.trim())
          ) {
            latestEntry.readyForPrompt = true;
          }
        }

      if (latestEntry.pendingPrompt && latestEntry.readyForPrompt && !latestEntry.promptSent) {
        latestEntry.promptSent = true;
        const text = latestEntry.pendingPrompt;
        // Send full prompt, then multiple enter presses to mimic a real submit
        terminalAPI.send({ id, data: text });
        setTimeout(() => terminalAPI.send({ id, data: '\r' }), 150);
        setTimeout(() => terminalAPI.send({ id, data: '\r' }), 450);
        setTimeout(() => terminalAPI.send({ id, data: '\r' }), 900);
      }

      if (
        !latestEntry.parentId &&
        latestEntry.readyForPrompt &&
        latestEntry.helperText &&
        !latestEntry.helperSent
      ) {
        latestEntry.helperSent = true;
        terminalAPI.send({ id, data: latestEntry.helperText });
        setTimeout(() => terminalAPI.send({ id, data: '\r' }), 180);
        setTimeout(() => terminalAPI.send({ id, data: '\r' }), 520);
      }
    }
  });

  terminalAPI.onExit((_, payload) => {
    if (payload.id !== id) return;
    if (!terminals.get(id)) return;
    stopThinking(entry);
    if (isChild && slot) {
      term.write(`\r\n[session exited ${payload.exitCode}]\r\n`);
      clearActiveTerminalIf(id);
      deselectEntry(entry);
      parkTerminal(entry);
      terminals.delete(id);
      childToSlot.delete(id);
      setSlotStatus(slot, null);
      collapseSlot(slot);
      removeSlot(slot);
      removeChildPanel(id);
      drawEdges();
      return;
    }
    term.write(`\r\n[session exited ${payload.exitCode}]\r\n`);
    clearActiveTerminalIf(id);
    deselectEntry(entry);
    parkTerminal(entry);
  });

  // Auto-scroll disabled per request
  const scrollToBottom = () => {};

  const scheduleFit = () => {
    if (!terminals.get(id)) return;
    if (entry.fitTimer) clearTimeout(entry.fitTimer);
    entry.fitTimer = setTimeout(() => {
      if (isTerminalInHost(entry)) {
        fitAndRefresh(entry);
      }
      drawEdges();
    }, 80);
  };

  if (!isChild) {
    // Dragging
    let dragging = false;
    let dragMoved = false;
    let dragOffset = { x: 0, y: 0 };
    const startDrag = (e) => {
      if (e.target.closest('.resizer')) return;
      if (e.target.closest('.master-summary')) return;
      dragging = true;
      dragMoved = false;
      const p = screenToWorld(e.clientX, e.clientY);
      dragOffset = { x: p.x - shell.offsetLeft, y: p.y - shell.offsetTop };
      e.preventDefault();
    };
    const onDrag = (e) => {
      if (!dragging) return;
      dragMoved = true;
      const p = screenToWorld(e.clientX, e.clientY);
      shell.style.left = `${p.x - dragOffset.x}px`;
      shell.style.top = `${p.y - dragOffset.y}px`;
      positionSlots(id);
      updateTitlePosition(id);
      scheduleFit();
      e.preventDefault();
    };
    const endDrag = () => {
      dragging = false;
      positionSlots(id);
      updateTitlePosition(id);
      scheduleFit();
    };
    if (header && header !== shell) header.addEventListener('mousedown', startDrag);
    shell.addEventListener('mousedown', startDrag);
    window.addEventListener('mousemove', onDrag);
    window.addEventListener('mouseup', endDrag);
    shell.addEventListener('click', (e) => {
      if (e.target.closest('.resizer')) return;
      if (dragMoved) {
        dragMoved = false;
        return;
      }
      setActiveTerminal(id);
      dragMoved = false;
    });

    drawEdges();

    // Initial fit after creation
    scheduleFit();

    // Resizing - keep circle shape
  } else {
    shell.classList.add('collapsed');
    drawEdges();
  }

  terminalCount += 1;
  return id;
}

function createMaster() {
  const now = Date.now();
  if (createMaster.lastCreatedAt && now - createMaster.lastCreatedAt < 500) {
    return;
  }
  createMaster.lastCreatedAt = now;
  const familyKey = pickFamilyKey();
  if (!familyKey) {
    alert('All flower families are currently in use. Close a manager to add a new one.');
    return;
  }
  const id = familyKey;
  // If somehow already used as an id, fall back to a suffixed id
  const uniqueId = terminals.has(id) ? `${id}-${Date.now().toString(36)}` : id;
  if (!familyAssignments.has(uniqueId)) {
    familyAssignments.set(uniqueId, familyKey);
    usedFamilies.add(familyKey);
  }
  createTerminal({ id: uniqueId, label: `${uniqueId} · manager` });
  ensureSlots(uniqueId);
  updateTitlePosition(uniqueId);
  const placeholder = document.getElementById('placeholder-shell');
  if (placeholder) {
    animatePlaceholderToCorner(placeholder);
  }
}

agentButtons.forEach((btn) => {
  btn.addEventListener('click', () => {
    const agentId = btn.dataset.agentOption;
    if (!agentId) return;
    setSelectedAgent(agentId);
  });
});
settingsButtons.forEach((btn) => btn.addEventListener('click', openSettings));
if (settingsCloseBtn) {
  settingsCloseBtn.addEventListener('click', closeSettings);
}
if (settingsOverlay) {
  settingsOverlay.addEventListener('click', (e) => {
    if (e.target === settingsOverlay) {
      closeSettings();
    }
  });
}
window.addEventListener('keydown', (e) => {
  if (e.key === 'Escape' && settingsOverlay && !settingsOverlay.classList.contains('hidden')) {
    closeSettings();
  }
});

refreshAgentDetection();
updateSidebarToggle();
updateCanvasCenterCache();

if (sidebarToggle) {
  sidebarToggle.addEventListener('click', toggleSidebar);
}
if (centerMasterBtn) {
  centerMasterBtn.addEventListener('click', centerOnOldestMaster);
}
if (controlPanel) {
  controlPanel.addEventListener('transitionend', (e) => {
    if (['width', 'max-width', 'min-width', 'padding', 'padding-left', 'padding-right'].includes(e.propertyName)) {
      updateCanvasCenterCache();
      scheduleRecenterOnLayout();
      refreshPlaceholderAnchor();
    }
  });
}

if (chooseFolderBtn) {
  chooseFolderBtn.addEventListener('click', chooseWorkingDirectory);
}
if (workingDirHelpBtn && workingDirTooltip) {
  workingDirHelpBtn.addEventListener('click', (e) => {
    e.stopPropagation();
    workingDirTooltip.classList.toggle('hidden');
  });
  document.addEventListener('click', (e) => {
    if (!workingDirTooltip.classList.contains('hidden') && !e.target.closest('#working-dir-help')) {
      workingDirTooltip.classList.add('hidden');
    }
  });
}
if (agentHelpBtn && agentTooltip) {
  agentHelpBtn.addEventListener('click', (e) => {
    e.stopPropagation();
    agentTooltip.classList.toggle('hidden');
  });
  document.addEventListener('click', (e) => {
    if (!agentTooltip.classList.contains('hidden') && !e.target.closest('#agent-help')) {
      agentTooltip.classList.add('hidden');
    }
  });
}

ipcRenderer.on('external-subagent', (_event, payload) => {
  const requestedMasterId = payload.master_agent_id;
  if (payload.child_id && terminals.has(payload.child_id)) {
    return;
  }
  const { id: parentId, note: parentNote } = pickOrCreateMaster(requestedMasterId);
  if (!parentId) return;
  ensureFamilyForMaster(parentId);
  const prompt = payload.prompt;
  const id = payload.child_id || allocateFlowerForMaster(parentId);
  ensureChildName(id, parentId);
  const label = `${id} · child of ${parentId}${parentNote || ''}`;
  const slot = findAvailableSlot(parentId);
  createTerminal({ id, label, parentId, prompt, slot });
  ipcRenderer.invoke('record-task', { masterId: parentId, childId: id, prompt });
});

ipcRenderer.on('external-title', (_event, payload) => {
  const masterId = payload?.master_agent_id;
  if (!masterId) return;
  if (payload?.title) {
    applyMasterTitle(masterId, payload.title, 'external');
  }
  if (typeof payload?.body === 'string') {
    applyMasterBody(masterId, payload.body, 'external');
  }
  if (typeof payload?.summary === 'string') {
    applyMasterSummary(masterId, payload.summary, 'external');
  }
  updateTitlePosition(masterId);
});

ipcRenderer.on('external-feedback', (_event, payload) => {
  // Feedback is logged to temp files; no auto-injection into child terminals
  if (payload.direction === 'to-child') {
    const childId = payload.child_id;
    const prompt = payload.prompt;
    const entry = terminals.get(childId);
    const slot = childToSlot.get(childId);
    if (!entry || !prompt) return;
    terminalAPI.send({ id: childId, data: prompt });
    setTimeout(() => terminalAPI.send({ id: childId, data: '\r' }), 120);
    if (slot) {
      setSlotStatus(slot, 'thinking');
      drawEdges();
    }
    logFeedback(childId, 'master-to-child', prompt);
  } else {
    // child -> master feedback received
    const childId = payload.child_id;
    const slot = childToSlot.get(childId);
    if (slot) {
      setSlotStatus(slot, 'update');
      drawEdges();
    }
    if (childId && payload.prompt) {
      logFeedback(childId, 'child-to-master', payload.prompt);
    }
  }
});

ipcRenderer.on('external-kill', (_event, payload) => {
  const childId = payload.child_id;
  if (!childId) return;
  const entry = terminals.get(childId);
  const reason = payload.reason;
  const slot = childToSlot.get(childId);
  if (!entry) return;
  const detail = reason ? `: ${reason}` : '';
  clearActiveTerminalIf(childId);
  deselectEntry(entry);
  parkTerminal(entry);
  entry.term.writeln(`\r\n[terminated by manager${detail}]`);
  terminalAPI.close({ id: childId });
  entry.term.dispose();
  terminals.delete(childId);
  childToSlot.delete(childId);
  const friendly = childNames.get(childId);
  if (friendly) usedFlowers.delete(friendly);
  childNames.delete(childId);
  if (slot) {
    setSlotStatus(slot, null);
    collapseSlot(slot);
    removeSlot(slot);
  } else if (entry.shell && entry.shell.parentNode) {
    entry.shell.parentNode.removeChild(entry.shell);
  }
  removeChildPanel(childId);
  drawEdges();
});

ipcRenderer.on('subagent-server-error', (_event, payload) => {
  const msg = payload?.message || 'Subagent server error';
  alert(msg);
});

function installPathForDisplay(file) {
  if (INSTALL_ENV_VAR) {
    const sep = process.platform === 'win32' ? '\\' : '/';
    const prefix = process.platform === 'win32' ? `%${INSTALL_ENV_VAR}%` : `$${INSTALL_ENV_VAR}`;
    const normalized = file.split(/[\\/]+/).join(sep);
    return `${prefix}${sep}${normalized}`;
  }
  return pathFromInstall(file);
}

function buildMasterHelperPrompt(masterId) {
  const template = loadMasterPromptTemplate();
  const scriptPath = installPathForDisplay('scripts/agents/spawn_subagent.js');
  const feedbackScript = installPathForDisplay('scripts/agents/submit_feedback.js');
  const killScript = installPathForDisplay('scripts/agents/kill_subagent.js');
  const titleScript = installPathForDisplay('scripts/agents/update_title.js');
  const statusScript = installPathForDisplay('scripts/agents/agent-status.js');
  const feedbackRoot = path.join(os.tmpdir(), 'agentgarden');
  const masterLabel = masterId || '<master_id>';
  const workingDir = getWorkingDirectory();
  return applyPromptTemplate(template, {
    WORKING_DIRECTORY: workingDir,
    MASTER_ID: masterLabel,
    SPAWN_SCRIPT: scriptPath,
    FEEDBACK_SCRIPT: feedbackScript,
    KILL_SCRIPT: killScript,
    TITLE_SCRIPT: titleScript,
    STATUS_SCRIPT: statusScript,
    FEEDBACK_ROOT: feedbackRoot,
  });
}

function loadMasterPromptTemplate() {
  try {
    if (fs.existsSync(MASTER_PROMPT_TEMPLATE_PATH)) {
      const stats = fs.statSync(MASTER_PROMPT_TEMPLATE_PATH);
      if (cachedMasterPromptTemplate && cachedMasterPromptMtime === stats.mtimeMs) {
        return cachedMasterPromptTemplate;
      }
      cachedMasterPromptTemplate = fs.readFileSync(MASTER_PROMPT_TEMPLATE_PATH, 'utf8');
      cachedMasterPromptMtime = stats.mtimeMs;
      masterPromptErrorLogged = false;
      return cachedMasterPromptTemplate;
    }
    if (!masterPromptErrorLogged) {
      console.warn(
        `Manager helper prompt template not found at ${MASTER_PROMPT_TEMPLATE_PATH}; using fallback text.`,
      );
      masterPromptErrorLogged = true;
    }
  } catch (err) {
    if (!masterPromptErrorLogged) {
      console.warn('Failed to load manager helper prompt template; using fallback text.', err);
      masterPromptErrorLogged = true;
    }
  }
  cachedMasterPromptTemplate = DEFAULT_MASTER_PROMPT_TEMPLATE;
  cachedMasterPromptMtime = null;
  return cachedMasterPromptTemplate;
}

function applyPromptTemplate(template, values) {
  if (!template) return '';
  return template.replace(/\{\{(\w+)\}\}/g, (match, key) => {
    if (Object.prototype.hasOwnProperty.call(values, key)) {
      return values[key];
    }
    return match;
  });
}

function refreshPromptHelper() {
  if (!promptText) return;
  const text = buildMasterHelperPrompt(lastMasterId);
  promptText.textContent = text;
  if (copyPromptBtn) {
    copyPromptBtn.onclick = async () => {
      try {
        await navigator.clipboard.writeText(text);
        copyPromptBtn.textContent = 'Copied';
        setTimeout(() => (copyPromptBtn.textContent = 'Copy prompt'), 1200);
      } catch (err) {
        alert('Could not copy to clipboard');
      }
    };
  }
}

function setupPromptHelper() {
  refreshPromptHelper();
}

setWorkingDirectory(selectedCwd);
setupPromptHelper();
refreshMasterEmptyState();
refreshChildEmptyState();
ensurePlaceholder();

canvas.addEventListener('wheel', handleWheelZoom, { passive: false, capture: true });
canvas.addEventListener('mousedown', (e) => {
  // ignore right/middle clicks
  if (e.button !== 0) return;
  handlePanStart(e);
});
window.addEventListener('mousemove', handlePanMove);
window.addEventListener('mouseup', handlePanEnd);

// Refocus and fit on window resize
window.addEventListener('resize', () => {
  updateCanvasCenterCache();
  terminals.forEach((entry) => {
    if (entry.fitTimer) clearTimeout(entry.fitTimer);
    const timer = setTimeout(() => {
      if (isTerminalInHost(entry)) {
        fitAndRefresh(entry);
      }
      drawEdges();
    }, 80);
    entry.fitTimer = timer;
  });
  masterSlots.forEach((_, masterId) => positionSlots(masterId));
  positionAllTitles();
  drawEdges();
  scheduleRecenterOnLayout();
  refreshPlaceholderAnchor();
});

window.addEventListener('fullscreenchange', () => {
  scheduleRecenterOnLayout();
  refreshPlaceholderAnchor();
});

function drawEdges() {
  // Clear existing SVG if any
  const old = document.getElementById('edges-layer');
  if (old) old.remove();

  const svg = document.createElementNS('http://www.w3.org/2000/svg', 'svg');
  svg.setAttribute('id', 'edges-layer');
  svg.style.position = 'absolute';
  svg.style.top = '0';
  svg.style.left = '0';
  svg.style.width = '100%';
  svg.style.height = '100%';
  svg.style.pointerEvents = 'none';

  masterSlots.forEach((slots, masterId) => {
    const master = terminals.get(masterId);
    if (!master) return;
    const mCircle = circleInfo(master.shell, 8);
    slots.forEach((slot) => {
      const sCircle = circleInfo(slot.element, 6);
      // Draw a simple center-to-center line between master and sub-agent circles
      const start = { x: sCircle.x, y: sCircle.y };
      const end = { x: mCircle.x, y: mCircle.y };
      const line = document.createElementNS('http://www.w3.org/2000/svg', 'line');
      line.setAttribute('x1', start.x);
      line.setAttribute('y1', start.y);
      line.setAttribute('x2', end.x);
      line.setAttribute('y2', end.y);
      line.classList.add('edge-line');
      if (!slot.occupiedBy) {
        line.classList.add('edge-empty');
      } else if (slot.status === 'thinking') {
        line.classList.add('edge-thinking');
      } else if (slot.status === 'waiting') {
        line.classList.add('edge-waiting');
      } else {
        line.classList.add('edge-connected');
      }
      svg.appendChild(line);
      if (slot.occupiedBy) {
        const leaf = document.createElementNS('http://www.w3.org/2000/svg', 'circle');
        const midX = (start.x + end.x) / 2 + (Math.random() * 10 - 5);
        const midY = (start.y + end.y) / 2 + (Math.random() * 10 - 5);
        leaf.setAttribute('cx', midX);
        leaf.setAttribute('cy', midY);
        leaf.setAttribute('r', '6.5');
        leaf.classList.add('edge-leaf');
        if (slot.status === 'thinking') {
          leaf.classList.add('edge-leaf-thinking');
        } else if (slot.status === 'waiting') {
          leaf.classList.add('edge-leaf-waiting');
        }
        svg.appendChild(leaf);
      }
    });
  });

  world.insertBefore(svg, world.firstChild);
  positionAllTitles();
}

document.addEventListener('click', (e) => {
  let insideExpanded = false;
  masterSlots.forEach((slots) => {
    slots.forEach((slot) => {
      if (slot.expanded && slot.element.contains(e.target)) {
        insideExpanded = true;
      }
    });
  });
  if (insideExpanded) return;
  masterSlots.forEach((slots) => {
    slots.forEach((slot) => {
      if (slot.expanded) {
        collapseSlot(slot);
      }
    });
  });
});

function ensureFeedbackMarker(childId) {
  const slot = childToSlot.get(childId);
  if (!slot || !slot.element) return null;
  if (feedbackMarkers.has(childId)) return feedbackMarkers.get(childId);
  const marker = document.createElement('button');
  marker.type = 'button';
  marker.className = 'feedback-marker';
  marker.setAttribute('aria-label', 'View feedback history');
  const icon = document.createElement('div');
  icon.className = 'feedback-icon';
  icon.innerHTML = '✉️';
  const dot = document.createElement('div');
  dot.className = 'feedback-dot';
  const tooltip = document.createElement('div');
  tooltip.className = 'feedback-tooltip tooltip-box hidden';
  marker.appendChild(icon);
  marker.appendChild(dot);
  marker.appendChild(tooltip);
  marker.addEventListener('click', (e) => {
    e.stopPropagation();
    toggleFeedbackTooltip(childId);
  });
  const entry = { marker, tooltip, childId };
  feedbackMarkers.set(childId, entry);
  return entry;
}

function toggleFeedbackTooltip(childId) {
  const entry = ensureFeedbackMarker(childId);
  if (!entry) return;
  const isOpen = entry.tooltip && !entry.tooltip.classList.contains('hidden');
  closeAllFeedbackTooltips(childId);
  if (!isOpen) {
    renderFeedbackTooltip(childId);
    entry.tooltip.classList.remove('hidden');
    entry.marker.classList.add('open');
  }
}

function closeAllFeedbackTooltips(exceptChildId = null) {
  feedbackMarkers.forEach((entry, id) => {
    if (exceptChildId && id === exceptChildId) return;
    entry.marker.classList.remove('open');
    entry.tooltip.classList.add('hidden');
  });
}

function closeAllTaskTooltips(exceptSlotId = null) {
  masterSlots.forEach((slots) => {
    slots.forEach((slot) => {
      if (exceptSlotId && slot.id === exceptSlotId) return;
      if (slot.taskTooltip) slot.taskTooltip.classList.add('hidden');
      if (slot.taskButton) slot.taskButton.classList.remove('open');
    });
  });
}

function renderFeedbackTooltip(childId) {
  const log = feedbackLogs.get(childId) || [];
  const entry = feedbackMarkers.get(childId) || ensureFeedbackMarker(childId);
  if (!entry || !entry.tooltip) return;
  entry.tooltip.innerHTML = '';
  if (!log.length) {
    entry.tooltip.textContent = 'No feedback yet';
    return;
  }
  const list = document.createElement('div');
  list.className = 'feedback-list';
  log.forEach((msg) => {
    const item = document.createElement('div');
    item.className = `feedback-item ${msg.direction === 'child-to-master' ? 'from-child' : 'from-manager'}`;
    const meta = document.createElement('div');
    meta.className = 'feedback-meta';
    const label = msg.direction === 'child-to-master' ? 'Child → Manager' : 'Manager → Child';
    meta.textContent = `${label} · ${formatTimeAgo(msg.ts)}`;
    const body = document.createElement('div');
    body.className = 'feedback-body';
    body.innerHTML = escapeHtml(msg.text || '');
    item.appendChild(meta);
    item.appendChild(body);
    list.appendChild(item);
  });
  entry.tooltip.appendChild(list);
  requestAnimationFrame(() => {
    entry.tooltip.scrollTop = entry.tooltip.scrollHeight;
  });
}

function logFeedback(childId, direction, text) {
  const existing = feedbackLogs.get(childId) || [];
  existing.push({ direction, text: text || '', ts: Date.now() });
  while (existing.length > 40) existing.shift();
  feedbackLogs.set(childId, existing);
  updateFeedbackMarker(childId);
}

function updateFeedbackMarker(childId) {
  const entry = ensureFeedbackMarker(childId);
  if (!entry) return;
  attachFeedbackMarker(childToSlot.get(childId));
  const hasLog = (feedbackLogs.get(childId) || []).length > 0;
  entry.marker.classList.toggle('visible', hasLog);
  renderFeedbackTooltip(childId);
}

document.addEventListener('click', (e) => {
  if (e.target.closest('.feedback-marker')) return;
  if (e.target.closest('.slot-task-button') || e.target.closest('.task-tooltip')) return;
  closeAllFeedbackTooltips();
  closeAllTaskTooltips();
});

const SCROLL_LOCK_SELECTOR =
  '.feedback-tooltip, .task-tooltip, .control-scroll, .master-panel-body, .sub-panel-body';

document.addEventListener(
  'wheel',
  (e) => {
    if (e.target.closest(SCROLL_LOCK_SELECTOR)) {
      e.stopPropagation();
    }
  },
  { capture: true },
);
